#ifndef FDC765_H_INCLUDED

#define FDC765_H_INCLUDED 1

/* 765: Library to emulate the uPD765a floppy controller (aka Intel 8272)

    Copyright (C) 2000  John Elliott <jce@seasip.demon.co.uk>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
#ifdef __cplusplus
extern "C" {
#endif

#ifndef NO_LIBDSK 
#include "libdsk.h"	
#endif
/*
    Functions which your program should provide are marked here by 
    the word EXT: 

    NOTE: This library emulates enough of a uPD765A controller (aka Intel 8272)
    for an Amstrad CP/M BIOS. Features the Amstrad BIOS does not use, such as:
 
    * DMA mode
    * Multitrack mode
    * Multisector reads/writes 
    * Read/Write deleted data

    are not implemented.

*/

#define MAX_SECTOR_LEN  8192	/* Max size of sectors this FDC can handle */
#define SHORT_TIMEOUT   1000	/* Timeouts, in machine cycles */
#define LONGER_TIMEOUT  1333333L

/* EXT: called by the FDC to print debugging messages
 * 
 * We assume that the normal debug level is 0. As we get more and more 
 * nitpicky, the debug level gets higher.
 *
 * Level 5 dumps out all command sequences sent to the FDC.
 * Level 4 logs all sector reads, writes & seeks.
 */
extern void fdc_dprintf(int debuglevel, char *fmt, ...);

/******************************* Types ************************************/
typedef unsigned char fdc_byte;	/* 8-bit byte */
typedef short         fd_err_t; /* Error code returned from disc routines */

/****************************** Defines ***********************************/

/* Floppy drive types */
#define FD_NC9256 (-1)	/* PCW9256 drives 2 and 3 */
#define FD_NONE   ( 0)	/* Drive B: on single-drive computers */
#define FD_30     ( 1)	/* 3"    */
#define FD_35     ( 2)	/* 3.5"  */
#define FD_525    ( 3)	/* 5.25" */

/* Floppy drive errors */
#define FD_E_OK	       (0) 	/* OK */
#define FD_E_SEEKFAIL (-1)	/* Seek fail */
#define FD_E_NOADDR   (-2)	/* Missing address mark */
#define FD_E_NODATA   (-3) 	/* No data */
#define FD_E_DATAERR  (-4)	/* CRC Data error */
#define FD_E_NOSECTOR (-5)	/* Sector not found */
#define FD_E_NOTRDY   (-6)	/* Drive not ready */
#define FD_E_READONLY (-7)	/* Read only */

struct fdc_765;

/********************************** CLASSES ********************************/

/* This is yet another attempt to write object-oriented C. 
 *
 * Variables in these structs are marked as PUBLIC, READONLY or PRIVATE. 
 * Your program can play with PUBLIC variables how it likes, but should
 * not alter READONLY ones, and it shouldn't touch the PRIVATE ones at all.
 *
 * Use FLOPPY_DRIVE for dummy drives (units which aren't connected at all) 
 * and DSK_FLOPPY_DRIVE / LIBDSK_FLOPPY_DRIVE for drives that are real.
 * 
 * These structures MUST be initialised (see fd_init() below) before use.
 */

typedef struct floppy_drive *FDRV_PTR;

typedef struct
{
	fd_err_t  (*fdv_seek_cylinder)(FDRV_PTR fd, int cylinder);
	fd_err_t  (*fdv_read_id)      (FDRV_PTR fd, int head, 
                                 int sector, fdc_byte *buf);
	fd_err_t  (*fdv_read_sector  )(FDRV_PTR fd, int xcylinder, 
		int xhead, int head, int sector, fdc_byte *buf, int len);
	fd_err_t  (*fdv_read_track   )(FDRV_PTR fd, int xcylinder,
		int xhead, int head, fdc_byte *buf, int *len);
	fd_err_t  (*fdv_write_sector )(FDRV_PTR fd, int xcylinder,
		int xhead, int head, int sector, fdc_byte *buf, int len);
	fd_err_t (*fdv_format_track )(FDRV_PTR fd, int head, 
		int sectors, fdc_byte *buf, fdc_byte filler);
	fdc_byte (*fdv_drive_status )(FDRV_PTR fd);
	int      (*fdv_isready)(FDRV_PTR fd);
	void     (*fdv_eject  )(FDRV_PTR fd);
} FLOPPY_DRIVE_VTABLE;


typedef struct floppy_drive
{
/* PUBLIC members */
	/* You should set the first three of these immediately after calling 
         * fd_init() or fdd_init() on a drive.*/
	int fd_type;	  /* 0 for none, 1 for 3", 2 for 3.5", 3 for 5.25" */
	int fd_heads;	  /* No. of heads in the drive: 1 or 2 */
	int fd_cylinders; /* No. of cylinders the drive can access: 
			   * eg: a nominally 40-track drive can usually go up
                           * to 42 tracks with a bit of "persuasion" */
	int fd_readonly;  /* Is the drive (or the disc therein) set to R/O? */

/* READONLY variables */
	int fd_motor;	  /* Is the motor for this drive running? */
	int fd_cylinder;  /* Current cylinder. Note that if the drive is
			   * double-stepping, this is the "real" cylinder - 
			   * so it could = 24 and be reading cylinder 12 
                           * of a 40-track DSK file. */
/* PRIVATE variables 
 * The following points to the drive's method table. You should not need
 * to use this; instead, use the fd_*() wrapper functions below. */
	FLOPPY_DRIVE_VTABLE * fd_vtable;	
} FLOPPY_DRIVE;

/* Subclass of FLOPPY_DRIVE: a drive which emulates discs using the CPCEMU 
 * .DSK format */

typedef struct dsk_floppy_drive
{
/* PUBLIC variables: */
	FLOPPY_DRIVE fdd;		/* Base class */
	char  fdd_filename[PATH_MAX];	/* Filename to .DSK file. Before 
					 * changing this call fd_eject() on
                                         * the drive */
/* PRIVATE variables: */
	FILE *fdd_fp;			/* File of the .DSK file */
	fdc_byte fdd_disk_header[256];	/* .DSK header */
	fdc_byte fdd_track_header[256];	/* .DSK track header */
} DSK_FLOPPY_DRIVE;

#ifndef NO_LIBDSK
typedef struct libdsk_floppy_drive
{
/* PUBLIC variables: */
	FLOPPY_DRIVE fdl;		/* Base class */
	char  fdl_filename[PATH_MAX];	/* Filename to .DSK file. Before 
					 * changing this call fd_eject() on
                                         * the drive */
	char  *fdl_type;		/* LIBDSK drive type, NULL for auto */
	DSK_PDRIVER fdl_diskp;	
	DSK_GEOMETRY fdl_diskg;		/* Autoprobed geometry */
} LIBDSK_FLOPPY_DRIVE;
#endif

typedef struct nc9_floppy_drive
{
	FLOPPY_DRIVE fdd;		/* Base class */
	FLOPPY_DRIVE *nc9_fdd;		/* Pointer to the 9256's B drive */
} NC9_FLOPPY_DRIVE;

/* This class represents the controller itself. When you instantiate one, call
 * fdc_reset() on it before doing anything else with it. */

typedef struct fdc_765
{
    /* PRIVATE variables */
	int fdc_interrupting;	/* 0 => Not interrupting
				 * 1 => Entering result phase of 
				 *      Read/Write/Format/Scan
				 * 2 => Ready for data transfer 
				 *      (execution phase) 
				 * 4 => End of Seek/Recalibrate command */
	/* The results from the SPECIFY command */
	int fdc_specify[2];
	/* The last sector for which a DD READ ID request was made */
	int fdc_lastidread; 

	/* Command phase buffer */
	int fdc_cmd_id;   	/* Current command */
	int fdc_cmd_len;  	/* No. of bytes remaining to transfer */
	int fdc_cmd_pos;	/* Next buffer position to write */
	fdc_byte fdc_cmd_buf[20];	/* The command as a byte string */

	/* Execution phase buffer */
	fdc_byte fdc_exec_buf[MAX_SECTOR_LEN];
	int  fdc_exec_len;	/* No. of bytes remaining to transfer */
	int  fdc_exec_pos;	/* Position in buffer */

	/* Results phase buffer */
	fdc_byte fdc_result_buf[20];
	int fdc_result_len;	/* No. of bytes remaining to transfer */	
	int fdc_result_pos;	/* Position in buffer */

	int fdc_terminal_count;	/* Set to abort a transfer */	
	int fdc_isr_countdown;	/* Countdown to interrupt */

	/* READONLY variables - these can be used in status displays */
        /* The four uPD765A status registers */
        int fdc_st0, fdc_st1, fdc_st2, fdc_st3;
        /* The main status register */
        int fdc_mainstat;
	int fdc_curunit, fdc_curhead;   /* Currently active unit & head */

	/* Public variables */
	void (*fdc_isr)(struct fdc_765 *self, int status);
		/* EXT: Called when interrupt line is raised or lowered.
		 *     You must provide this if the FDC is to interrupt. */
	FLOPPY_DRIVE *fdc_drive[4];
		/* The FDC's four drives. You must set these pointers */

} FDC_765;


/*********************** WRAPPER FUNCTIONS ********************************/ 

/* Wrapper functions for drives */

/* Initialise a FLOPPY_DRIVE as a dummy (eg:disconnected B drive) */
void fd_init(FLOPPY_DRIVE *fd);
/* Initialise a FLOPPY_DRIVE as a 9256 dummy (for PCW9256 drives 2 & 3) */
void fd_init_nc9256(NC9_FLOPPY_DRIVE *nc9);
/* Initialise a DSK_FLOPPY_DRIVE as a full drive */
void fdd_init(DSK_FLOPPY_DRIVE *fdd);
/* As fdd_init, but uses LIBDSK */
void fdl_init(LIBDSK_FLOPPY_DRIVE *fdd);

/* Wrappers for the floppy's member functions */
/* Seek to a cylinder */
fd_err_t fd_seek_cylinder(FLOPPY_DRIVE *fd, int cylinder);
/* Read ID from current cylinder, head "head". "sector" is a number 
 * which should increase every time you call this function, to simulate 
 * the rotation of the disc. Fills the buffer at "buf" with the sector ID. */
fd_err_t fd_read_id(FLOPPY_DRIVE *fd, int head, int sector, fdc_byte *buf);
/* Read a sector. xcylinder and xhead are values expected from the sector ID;
 * while "head" and "sector" are the actual place to look on the disc. 
 * Data will be returned to "buf", maximum "len" bytes. */
fd_err_t fd_read_sector(FLOPPY_DRIVE *fd, int xcylinder,
                int xhead, int head, int sector, fdc_byte *buf, int len);
/* Write a sector; parameters as for "read sector" */
fd_err_t  fd_write_sector(FLOPPY_DRIVE *fd, int xcylinder,
                int xhead, int head, int sector, fdc_byte *buf, int len);
/* Read an entire track; parameters as for "read sector" */
fd_err_t  fd_read_track   (struct floppy_drive *fd, int xcylinder,
                int xhead, int head, fdc_byte *buf, int *len);
/* Format a track of "sectors" sectors, on the current cylinder. "track" 
 * holds the sector ID information, and "filler" is the filler byte */ 
fd_err_t  fd_format_track (struct floppy_drive *fd, int head, 
		int sectors, fdc_byte *track, fdc_byte filler);
/* Get the drive status - bits 7-3 of the DD_DRIVE_STATUS value */
fdc_byte fd_drive_status(FLOPPY_DRIVE *fd);
/* Return 1 if the drive is ready, 0 if it is not */
fdc_byte fd_isready(FLOPPY_DRIVE *fd);
/* Eject the disc from the drive */
void fd_eject(FLOPPY_DRIVE *fd);

/* This function is specific to DSK-type drives, and is not called by the 
 * FDC. It is intended for use by administration interfaces */
fd_err_t fdd_new_dsk(DSK_FLOPPY_DRIVE *fd);


/* Interface to the FDC itself */ 

/* Reset the FDC. MUST be called to initialise the FDC structure */
void fdc_reset     (FDC_765 *self);
/* Write a byte to the FDC's data register */
void fdc_write_data(FDC_765 *self, fdc_byte value);
/* Read the FDC's data register */
fdc_byte fdc_read_data (FDC_765 *self);
/* Read the FDC's main control register */
fdc_byte fdc_read_ctrl (FDC_765 *self);
/* Raise / lower the FDC's terminal count line */
void fdc_set_terminal_count(FDC_765 *self, fdc_byte value);
/* Start/stop drive motors. Bit 0 corresponds to drive 1's motor, etc. */
void fdc_set_motor(FDC_765 *self, fdc_byte running);
/* Call this once every cycle round the emulator's main loop */
void fdc_tick(FDC_765 *self);

#ifdef __cplusplus
}
#endif                                                                          

#endif /* ndef FDC765_H_INCLUDED */

