/* 765: Library to emulate the uPD765a floppy controller (aka Intel 8272)

    Copyright (C) 2000  John Elliott <jce@seasip.demon.co.uk>

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/
#include <stdio.h>
#include <limits.h>
#include "765.h"

/*
 * Default implementations of the FDC functions. If the pointer to the drive
 * or the function pointer is null, it returns results as if the drive 
 * were not present.
 */

/* Seek to a cylinder */
fd_err_t fd_seek_cylinder(FLOPPY_DRIVE *fd, int cylinder)
{
	if (fd && (fd->fd_vtable->fdv_seek_cylinder))
	{
		return (*fd->fd_vtable->fdv_seek_cylinder)(fd, cylinder);
	}
	return FD_E_NOTRDY;
}


/* Read the ID of the next sector to pass under the head. "sector" is 
 * suggested since most emulated drives don't actually emulate the idea
 * of a head being over one sector at a time */
fd_err_t  fd_read_id(FLOPPY_DRIVE *fd, int head, int sector, fdc_byte *buf)
{
	if (fd && (fd->fd_vtable->fdv_read_id))
	{
		return (*fd->fd_vtable->fdv_read_id)(fd, head, sector, buf);
	}
	return FD_E_NOTRDY;
}

/* Read a sector. xcylinder and xhead are the expected values for the 
 * sector header; head is the actual head to use. */
fd_err_t  fd_read_sector(FLOPPY_DRIVE *fd, int xcylinder, 
		int xhead, int head, int sector, fdc_byte *buf, int len)
{
	if (fd && (fd->fd_vtable->fdv_read_sector))
	{
		return (*fd->fd_vtable->fdv_read_sector)(fd, xcylinder, 
			xhead, head, sector, buf, len);
	}
	return FD_E_NOTRDY;
}

/* Read a track. xcylinder and xhead are the expected values for the 
 * sector header; head is the actual head to use. */
fd_err_t  fd_read_track(FLOPPY_DRIVE *fd, int xcylinder,
                int xhead, int head, fdc_byte *buf, int *len)
{
        if (fd && (fd->fd_vtable->fdv_read_track))
        {
                return (*fd->fd_vtable->fdv_read_track)(fd, xcylinder,
                        xhead, head, buf, len);
        }
        return FD_E_NOTRDY;
}



/* Write a sector. xcylinder and xhead are the expected values for the 
 * sector header; head is the actual head to use. */
fd_err_t  fd_write_sector(FLOPPY_DRIVE *fd, int xcylinder,
                int xhead, int head, int sector, fdc_byte *buf, int len)
{
        if (fd && (fd->fd_vtable->fdv_write_sector))
        {
                return (*fd->fd_vtable->fdv_write_sector)(fd, xcylinder, 
                        xhead, head, sector, buf, len);
        }
        return FD_E_NOTRDY;
}

/* Format a track */
fd_err_t  fd_format_track (struct floppy_drive *fd, int head,
                int sectors, fdc_byte *track, fdc_byte filler)
{
	if (fd && (fd->fd_vtable->fdv_format_track))
	{
		return (*fd->fd_vtable->fdv_format_track)(fd, head, sectors, track, filler);
	}
	return FD_E_NOTRDY;
}


/* Get the drive status (as given in bits 7-3 of DD_DRIVE_STATUS) */
fdc_byte fd_drive_status(FLOPPY_DRIVE *fd)
{
	if (fd && (fd->fd_vtable->fdv_drive_status))
	{
		return (*fd->fd_vtable->fdv_drive_status)(fd);
	}
	return 0;	/* Drive not present */
}


/* Is the drive ready? */
fdc_byte fd_isready(FLOPPY_DRIVE *fd)
{
	if (fd && (fd->fd_vtable->fdv_isready)) return (*fd->fd_vtable->fdv_isready)(fd);
	return 0;
}

/* Eject under computer's control */
void fd_eject(FLOPPY_DRIVE *fd)
{
	if (fd && (fd->fd_vtable->fdv_eject)) 
		(*fd->fd_vtable->fdv_eject)(fd);
}


/* On the PCW9256, drives 2 and 3 always return ready. Drive 2 at least
 * passes all its other commands to drive 1. */

static fd_err_t n9256_seek_cylinder(FLOPPY_DRIVE *fd, int cylinder)
{
        NC9_FLOPPY_DRIVE *nc9 = (NC9_FLOPPY_DRIVE *)fd;
        if (nc9->nc9_fdd) return fd_seek_cylinder(nc9->nc9_fdd, cylinder);

	return FD_E_NOTRDY;
}


static fdc_byte n9256_drive_status(FLOPPY_DRIVE *fd)
{
	fdc_byte b = 0;
	
	NC9_FLOPPY_DRIVE *nc9 = (NC9_FLOPPY_DRIVE *)fd;
	if (nc9->nc9_fdd) b = fd_drive_status(nc9->nc9_fdd);

	return 0x20 | b;	/* Drive is always ready */
}


static FLOPPY_DRIVE_VTABLE dummy_vtbl;	/* all NULLs */
static FLOPPY_DRIVE_VTABLE d9256_vtbl =	/* nearly all NULLs */
{
	n9256_seek_cylinder,
	NULL,
	NULL,
	NULL,
	NULL,
	NULL,
        n9256_drive_status,
	NULL,
	NULL
};

/* Initialise a FLOPPY_DRIVE structure */
void fd_init(FLOPPY_DRIVE *fd)
{
	fd->fd_type      = FD_NONE;
	fd->fd_heads     = 0;
	fd->fd_cylinders = 0;
	fd->fd_motor     = 0;
	fd->fd_cylinder  = 0;
	fd->fd_readonly  = 0;	
	fd->fd_vtable    = &dummy_vtbl;
}

/* Initialise a 9256 dummy drive */
void fd_init_nc9256(NC9_FLOPPY_DRIVE *nc9)
{
	fd_init(&nc9->fdd);
	nc9->fdd.fd_type = FD_NC9256;
	nc9->nc9_fdd     = NULL;
//
// These are the only commands which CP/M executes on a 9256 dummy drive,
// and so these are the only ones I'm going to pass through to the 
// underlying drive.
//
	nc9->fdd.fd_vtable    = &d9256_vtbl;
}


