/************************************************************************

    JOYCE v1.90 - Amstrad PCW emulator

    Copyright (C) 1996, 2001  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/

#include "Joyce.hxx"
#include "UiContainer.hxx"
#include "UiLabel.hxx"
#include "UiSeparator.hxx"
#include "UiCommand.hxx"
#include "UiSetting.hxx"
#include "UiMenu.hxx"
#include "UiTextInput.hxx"
#include "UiScrollingMenu.hxx"
#include "JoyceFileList.hxx"
#include "JoyceFileChooser.hxx"


JoyceFDC::JoyceFDC(JoyceSystem *s) : JoyceDevice("fdc")
{
	int n;
	XLOG("JoyceFDC::JoyceFDC()");

	m_sys = s;

	for (n = 0; n < 4; n++)
	{
		m_driveType[n] = FD_35;
		m_driveHeads[n] = 2;
		m_driveCyls[n] = 83;
		m_driveMap[n] = (n % 2);
	}
	m_driveMap[2] = 10;	// nc9 model
	m_driveMap[3] = 11;
	m_bootRO = false;
	
	reset();	
}

JoyceFDC::~JoyceFDC()
{
        for (int n = 0; n < 4; n++)
        {
                fd_eject(m_fdc.fdc_drive[n]);
        }

}

static JoyceFDC *theFDC;

static void JoyceFdcIsr(FDC_765 *fdc, int status)
{
	theFDC->isr(fdc, status);
}


void JoyceFDC::isr(FDC_765 *fdc, int status)
{
        if (m_fdcInt != status)
	{	
//                joyce_dprintf("FDC interrupt goes to %d\n", status);
		m_fdcInt = status;
	}
        m_sys->m_asic.fdcIsr(status);
}


void JoyceFDC::setTerminalCount(bool b)
{
	m_terminalCount = b;
	fdc_set_terminal_count(&m_fdc, b);
}

void JoyceFDC::setMotor(byte b)
{
	fdc_set_motor(&m_fdc, b);
}


void JoyceFDC::writeData(byte b)
{
	fdc_write_data(&m_fdc, b); 
}


byte JoyceFDC::readData(void)
{
	return fdc_read_data(&m_fdc);
}


byte JoyceFDC::readControl(void)
{
	return fdc_read_ctrl(&m_fdc);
}


void JoyceFDC::tick(void)
{
        fdc_tick(&m_fdc);  /* Check for end of FDC operation */
}


void JoyceFDC::insertB(const string filename)
{
	int err;

	//fd_eject(&m_fd[1].fdd);
	//strcpy(m_fd[1].fdd_filename, filename.c_str());
	fd_eject(&m_fdl[1].fdl);
	strcpy(m_fdl[1].fdl_filename, filename.c_str());
	m_fdl[1].fdl_type = NULL;
}



bool JoyceFDC::loadBootSector(const string filename, char *type, byte *sector)
{
	int err;

	//fd_eject(&m_fd[0].fdd);
	//strcpy(m_fd[0].fdd_filename, filename.c_str());
        //joyce_dprintf("Loading %s\n", m_fd[0].fdd_filename);
	//m_fd[0].fdd.fd_readonly = m_bootRO;
	fd_eject(&m_fdl[0].fdl);
	strcpy(m_fdl[0].fdl_filename, filename.c_str());
	m_fdl[0].fdl_type = type;
        joyce_dprintf("Loading %s\n", m_fdl[0].fdl_filename);
	m_fdl[0].fdl.fd_readonly = m_bootRO;

	setMotor(0x0F);

       // if (!fd_isready(&m_fd[0].fdd)) return 0;
        if (!fd_isready(&m_fdl[0].fdl)) return 0;

        joyce_dprintf("Image file exists.\n");

        //err = fd_seek_cylinder(&m_fd[0].fdd, 0);
        err = fd_seek_cylinder(&m_fdl[0].fdl, 0);

        if (err)
        {
                joyce_dprintf("Seek error while booting: %d\n", err);
                return false;       /* Track 0 */
        }
        //err = fd_read_sector(&m_fd[0].fdd, 0,0,0,1, sector, 512);
        err = fd_read_sector(&m_fdl[0].fdl, 0,0,0,1, sector, 512);
        if (err)
        {
                joyce_dprintf("Read error while booting: %d\n", err);
                return false;       /* Could not read sector */
        }
	return true;
}





void JoyceFDC::reset(void)
{
	// eject all drives (thus, hopefully, committing changes)
        for (int n = 0; n < 4; n++)
        {
                fd_eject(m_fdc.fdc_drive[n]);
        }
	reset(0);
}

void JoyceFDC::reset(int somedrives)
{
	int n, idx;

	if (!somedrives)
	{
		theFDC = this;
		m_fdcInt = false;
		fdc_reset(&m_fdc);
		m_fdc.fdc_isr = JoyceFdcIsr;
		m_terminalCount = false;
	}
	for (n = somedrives; n < 4; n++)
	{
                //fdd_init      (&m_fdd[n]);
                fdl_init      (&m_fdl[n]);
                fd_init       (&m_fd_none[n]);
                fd_init_nc9256(&m_fd_9256[n]);

                m_fdl[n].fdl.fd_type  = m_driveType[n];
                m_fdl[n].fdl.fd_heads = m_driveHeads[n];
                m_fdl[n].fdl.fd_cylinders = m_driveCyls[n];

		idx = m_driveMap[n];
		switch(idx)
		{
			case 0: case 1: case 2: case 3:
        		//m_fdc.fdc_drive[n] = &m_fd[idx].fdd; break;
        		m_fdc.fdc_drive[n] = &m_fdl[idx].fdl; break;
			case 4: case 5: case 6: case 7:
        		m_fdc.fdc_drive[n] = &m_fd_none[idx - 4]; break;
			case 8: case 9: case 10: case 11:
        		m_fdc.fdc_drive[n] = &m_fd_9256[idx - 8].fdd; break;
			case 12: case 13: case 14: case 15:
			m_fdc.fdc_drive[n] = m_fdc.fdc_drive[idx - 12]; break;
		}	
	}
	// Make the 9256 disconnected drives 2,3 use unit 1 as their proxy.
	for (n = 2; n < 4; n++) m_fd_9256[n].nc9_fdd = m_fdc.fdc_drive[1];
}


// Parsing XML configuration data...

bool JoyceFDC::parseNode(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
        int bootro = 0;
        
	// Boot drive R/O setting
	char *s = (char *)xmlGetProp(cur, (xmlChar *)"bootro");

        if      (s && !strcmp(s,"1")) bootro = 1;
        else if (s && strcmp(s,"0")) 
		joyce_dprintf("FDC boot read-only must be 0 or 1 - not %s\n", s);
        if (s) free(s);

        m_bootRO = bootro;

	// Settings for individual drives
	cur = cur->xmlChildrenNode;
	while (cur != NULL)
	{
		if (!strcmp((char *)cur->name, "drive") && (cur->ns == ns))
		{
			if (!parseDriveNode(doc, ns, cur)) return false;
		}
		cur = cur->next;
	}
	return true;
}



bool JoyceFDC::parseDriveNode(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
	char *s, *t, *type, *cyls, *heads;
	int id = -1, model = -1, as = -1;

	s = (char *)xmlGetProp(cur, (xmlChar *)"id");
        if (!s || s[1] != 0 || s[0] < '0' || s[0] > '3')
	{
        	joyce_dprintf("<drive> must have an id= field\n");
		return true;
	}
	id = atoi(s);
	if (s) free(s);	

	s = (char *)xmlGetProp(cur, (xmlChar *)"model");
	if (s && !strcmp(s, "dsk")) model = 1;
	if (s && !strcmp(s, "nc8")) model = 2;
	if (s && !strcmp(s, "nc9")) model = 3;
	if (s && !strcmp(s, "same")) model = 4;

	if (model < 1) joyce_dprintf("<drive> %d has unrecognised "
					"model type %s\n", s);
	else switch(model)
	{
		case 1: /* real drive */
		type  = (char *)xmlGetProp(cur, (xmlChar *)"type");
		cyls  = (char *)xmlGetProp(cur, (xmlChar *)"cylinders");
		heads = (char *)xmlGetProp(cur, (xmlChar *)"heads");
		m_fdc.fdc_drive[id] = &m_fdl[id].fdl;
		m_driveMap[id] = id;
		//fdd_init(&m_fd[id]);
		fdl_init(&m_fdl[id]);

		if (cyls) 
		{ 
			m_driveCyls[id] = 
			m_fdl[id].fdl.fd_cylinders = atoi(cyls);
		}
		if (heads) 
		{ 
			m_driveHeads[id] = 
			m_fdl[id].fdl.fd_heads = atoi(heads);
		}
		if (type)
		{
			if (!strcmp(type, "3.5"))  
				m_driveType[id] = m_fdl[id].fdl.fd_type = FD_35;	
			if (!strcmp(type, "5.25")) 
				m_driveType[id] = m_fdl[id].fdl.fd_type = FD_525;	
			if (!strcmp(type, "3.0"))  
				m_driveType[id] = m_fdl[id].fdl.fd_type = FD_30;	
		}
		if (type)  free(type);
		if (heads) free(heads);
		if (cyls)  free(cyls);
		break;

		case 2: /* nc8 */
		m_driveMap[id] = id + 4;
                m_fdc.fdc_drive[id] = &m_fd_none[id];
                fd_init(&m_fd_none[id]);
                m_fd_none[id].fd_type = FD_NONE;
		break;

		case 3:	/* nc9 */
		m_driveMap[id] = id + 8;
		m_fdc.fdc_drive[id] = &m_fd_9256[id].fdd;
		fd_init_nc9256(&m_fd_9256[id]);
		m_fd_9256[id].fdd.fd_type = FD_NC9256;
		break;

		case 4: /* Same as */
		t = (char *)xmlGetProp(cur, (xmlChar *)"ref");
		if (t) 
		{ 
			as = atoi(t);
			free(t);
		}
		if (as >= 0 && as <= 3) m_driveMap[id] = 12 + as;
		m_fdc.fdc_drive[id] = m_fdc.fdc_drive[as];
		break;

	}
	if (s) free(s);
	return true;
}



bool JoyceFDC::storeNode(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
	bool some;
	char buf[20];
	xmlNodePtr drv;

	xmlSetProp(cur, (xmlChar *)"bootro", (xmlChar *)(m_bootRO ? "1" : "0"));

	for (int n = 0; n < 4; n++)
	{
		some = false;
                sprintf(buf, "%d", n);
		// Search for existing matching <drive> node.
                for (drv = cur->xmlChildrenNode; drv; drv = drv->next)
                {
			if (strcmp((char *)drv->name, "drive")) continue;

			char *s = (char *)xmlGetProp(drv, (xmlChar *)"id");
			if (!s) continue;
			if (strcmp(s, buf)) 
			{
				free(s); 
				continue;
			}
			some = true;
			storeDriveNode(doc, ns, drv, n);
                }
		// No existing <drive> node; create one.
		if (!some)
		{
			drv = xmlNewNode(ns, (xmlChar *)"drive");
			xmlSetProp(drv, (xmlChar *)"id", (xmlChar *)buf);
			xmlAddChild(cur, drv);
			storeDriveNode(doc, ns, drv, n);
		}
	}
	return true;
}


bool JoyceFDC::storeDriveNode(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur, int n)
{
	char buf[20];

	xmlSetProp(cur, (xmlChar *)"cylinders", NULL);
	xmlSetProp(cur, (xmlChar *)"heads",     NULL);
	xmlSetProp(cur, (xmlChar *)"type",      NULL);
	xmlSetProp(cur, (xmlChar *)"ref",       NULL);

	switch(m_driveMap[n])
	{
		case  0: case  1: case  2: case  3:
		xmlSetProp(cur, (xmlChar *)"model", (xmlChar *)"dsk");
		sprintf(buf, "%d", m_driveCyls[n]);
		xmlSetProp(cur, (xmlChar *)"cylinders", (xmlChar *)buf);
                sprintf(buf, "%d", m_driveHeads[n]);
                xmlSetProp(cur, (xmlChar *)"heads", (xmlChar *)buf);
                switch(m_driveType[n])
                {
                	case FD_35:  xmlSetProp(cur, (xmlChar *)"type", (xmlChar *)"3.5");  break;
			case FD_30:  xmlSetProp(cur, (xmlChar *)"type", (xmlChar *)"3.0");  break;
			case FD_525: xmlSetProp(cur, (xmlChar *)"type", (xmlChar *)"5.25"); break;
		}
		break;	

		case  4: case  5: case  6: case  7:
		xmlSetProp(cur, (xmlChar *)"model", (xmlChar *)"nc8");
		break;

		case  8: case  9: case 10: case 11:
		xmlSetProp(cur, (xmlChar *)"model", (xmlChar *)"nc9");
		xmlSetProp(cur, (xmlChar *)"ref",   (xmlChar *)"1");
		break;

		case 12: case 13: case 14: case 15:
		sprintf(buf, "%d", m_driveMap[n] - 12);
		xmlSetProp(cur, (xmlChar *)"model", (xmlChar *)"same");
		xmlSetProp(cur, (xmlChar *)"ref",   (xmlChar *)buf);
		break;
	}
	return true;
}



bool JoyceFDC::hasSettings(SDLKey *key, string &caption)
{
        *key = SDLK_d;
        caption = "  Disc drives  ";
        return true;
}

UiEvent JoyceFDC::settings(UiDrawer *d)
{
	UiEvent uie;
	bool is8256, is9256, advanced;
	int sel = -1;
        int driveType[4];
        int driveHeads[4];
        int driveCyls[4];
        int driveMap[4];
	int n, changes;

	for (n = 0; n < 4; n++) 
	{
		driveType[n]  = m_driveType[n];
		driveHeads[n] = m_driveHeads[n];
		driveCyls[n]  = m_driveCyls[n];
		driveMap[n]   = m_driveMap[n];
	}
	
	while (1)
	{
		UiMenu uim(d);

		is9256 = is8256 = advanced = false;
		if (m_driveMap[2] == 10 && m_driveMap[3] == 11) is9256 = true;
		if (m_driveMap[2] == 12 && m_driveMap[3] == 13) is8256 = true; 
		// Drives C and D are mapped in a nonstandard way...
		if (is8256 == 0 && is9256 == 0) advanced = true;

		uim.add(new UiLabel  ("         Disc drives         ", d));
		uim.add(new UiSeparator(d));
		uim.add(new UiSetting(SDLK_w, !m_bootRO, "  Boot discs are read/Write  ",  d));
		uim.add(new UiSetting(SDLK_o,  m_bootRO, "  Boot discs are read-Only   ",  d));
		uim.add(new UiSeparator(d));
		uim.add(new UiSetting(SDLK_8, is8256, "  8256/8512/9512 controller  ",  d));
		uim.add(new UiSetting(SDLK_9, is9256, "  9256/9512+/10  controller  ",  d));
		uim.add(new UiSeparator(d));
		if (advanced) 
		{
			uim.add(new UiCommand(SDLK_0, "  Drive 0 settings... ", d, UIG_SUBMENU));	
			uim.add(new UiCommand(SDLK_1, "  Drive 1 settings... ", d, UIG_SUBMENU));	
			uim.add(new UiCommand(SDLK_2, "  Drive 2 settings... ", d, UIG_SUBMENU));	
			uim.add(new UiCommand(SDLK_3, "  Drive 3 settings... ", d, UIG_SUBMENU));	
		}
		else
		{
			uim.add(new UiCommand(SDLK_a, "  Drive A: settings... ", d, UIG_SUBMENU));	
			uim.add(new UiCommand(SDLK_b, "  Drive B: settings... ", d, UIG_SUBMENU));	
		}
		uim.add(new UiSeparator(d));
		uim.add(new UiCommand(SDLK_ESCAPE, "  EXIT  ", d));
		d->centreContainer(uim);
		uim.setSelected(sel);
		uie = uim.eventLoop();

		if (uie == UIE_QUIT) return uie;
		switch(uim.getKey(sel = uim.getSelected()))
		{
			case SDLK_w: m_bootRO = false; break;
			case SDLK_o: m_bootRO = true;  break;
			case SDLK_9: m_driveMap[2] = 10; m_driveMap[3] = 11; break;
			case SDLK_8: m_driveMap[2] = 12; m_driveMap[3] = 13; break;
			case SDLK_0:	
			case SDLK_a: uie = driveSettings(0, advanced, d); 
				     if (uie == UIE_QUIT) return uie;
				     break;
			case SDLK_1:	
			case SDLK_b: uie = driveSettings(1, advanced, d); 
				     if (uie == UIE_QUIT) return uie;
				     break;
			case SDLK_2: uie = driveSettings(2, advanced, d); 
				     if (uie == UIE_QUIT) return uie;
				     break;
			case SDLK_3: uie = driveSettings(3, advanced, d); 
				     if (uie == UIE_QUIT) return uie;
				     break;

			case SDLK_ESCAPE:
//
// See if settings have changed. If so, rewire the FDC. 
//
					changes = 0;
					for (n = 0; n < 4; n++) 
					{
						changes |= (driveType[n]  != m_driveType[n]);
		                                changes |= (driveHeads[n] != m_driveHeads[n]);
		                                changes |= (driveCyls[n]  != m_driveCyls[n]);
		                                changes |= (driveMap[n]   != m_driveMap[n]);
					}
					if (changes)
					{
        					for (n = 0; n < 4; n++)
                					fd_eject(m_fdc.fdc_drive[n]);
						reset();
						// Update the PCW type in JoyceSystem.
						m_sys->setModel(adaptModel(m_sys->getModel()));
					}
					return UIE_OK;
			default:	break;
		}	// end switch
	}		// end while
	return UIE_OK;
}

UiEvent JoyceFDC::driveSettings(int unit, bool advanced, UiDrawer *d)
{
	char buf[40];
	int sel = -1;
	bool fd1, fd2, fd35, fdother, fdnc9, fdnc8, fdsameas;
	UiEvent uie;

	while (1)
	{	
		fd1 = fd2 = fd35 = fdother = fdnc9 = fdnc8 = fdsameas = false;
		if (m_driveMap[unit] < 4)
		{
			if (m_driveType[unit] == FD_30 && m_driveHeads[unit] == 1 &&
			    m_driveCyls[unit] >= 40 && m_driveCyls[unit] <= 43) fd1 = true;
			if (m_driveType[unit] == FD_30 && m_driveHeads[unit] == 2 &&
			    m_driveCyls[unit] >= 80 && m_driveCyls[unit] <= 86) fd2 = true;
			if (m_driveType[unit] == FD_35 && m_driveHeads[unit] == 2 &&
			    m_driveCyls[unit] >= 80 && m_driveCyls[unit] <= 86) fd35 = true;
		}
		if (m_driveMap[unit] >= 4  && m_driveMap[unit] < 8) fdnc8 = true;
		if (advanced)
		{
			if (m_driveMap[unit] >= 8  && m_driveMap[unit] < 12) fdnc9 = true;
			if (unit > 0 && m_driveMap[unit] >= 12 && m_driveMap[unit] < 16) fdsameas = true;
		}
		if (!fd35 && !fd2 && !fd1 && !fdnc9 && !fdnc8	
			&& !fdsameas) fdother = true;

		UiMenu uim(d);
	
		if (advanced) sprintf(buf, "   Unit %d settings   ", unit);
		else	      sprintf(buf, "  Drive %c: settings  ", unit + 'A');
	
		uim.add(new UiLabel(buf, d));
		uim.add(new UiSeparator(d));
		if (unit || advanced) 
			uim.add(new UiSetting(SDLK_n, fdnc8,   "  Not connected     ", d));
		uim.add(new UiSetting(SDLK_1, fd1,     "  180k 3\" drive    ", d));
		uim.add(new UiSetting(SDLK_2, fd2,     "  720k 3\" drive    ", d));
		uim.add(new UiSetting(SDLK_3, fd35,    "  720k 3.5\" drive  ", d));
		uim.add(new UiSetting(SDLK_o, fdother, "  Other...          ", d));
		if (advanced)
		{
			sprintf(buf, "  Same as unit %d  ", m_driveMap[unit] % 4);
			uim.add(new UiSetting(SDLK_9, fdnc9,    "  9256 not connected  ", d));
			if (unit > 0) uim.add(new UiSetting(SDLK_s, fdsameas, buf, d));
		}
		uim.add(new UiSeparator(d));
		uim.add(new UiCommand(SDLK_ESCAPE, "  EXIT  ", d));	
		d->centreContainer(uim);
		uim.setSelected(sel);
		uie = uim.eventLoop();
		if (uie == UIE_QUIT) return uie;
		sel = uim.getSelected();
		switch(uim.getKey(sel))
		{
			case SDLK_n: m_driveMap [unit] = unit + 4;
				     m_driveType[unit] = FD_NONE;
				     break;
			case SDLK_9: m_driveMap [unit] = unit + 8;
				     m_driveType[unit] = FD_NC9256;
				     break;
			case SDLK_1: m_driveMap  [unit] = unit;
				     m_driveType [unit] = FD_30;
		                     m_driveHeads[unit] = 1;
				     m_driveCyls [unit] = 43;
				     break;
			case SDLK_2: m_driveMap  [unit] = unit;
				     m_driveType [unit] = FD_30;
		                     m_driveHeads[unit] = 2;
				     m_driveCyls [unit] = 83;
				     break;
			case SDLK_3: m_driveMap  [unit] = unit;
				     m_driveType [unit] = FD_35;
		                     m_driveHeads[unit] = 2;
				     m_driveCyls [unit] = 83;
				     break;
			case SDLK_o:	// Set up custom geometry
				    uie = driveCustom(unit, advanced, d);
				    if (uie == UIE_QUIT) return uie;
				    break;
			case SDLK_a:	// Same as
				    uie = driveSameAs(unit, d);
				    if (uie == UIE_QUIT) return uie;	
				    break;
			case SDLK_ESCAPE: return UIE_CONTINUE;
			default:	break;
		}
	}	
	return UIE_CONTINUE;
}


UiEvent JoyceFDC::driveCustom(int unit, bool advanced, UiDrawer *d)
{
	int sel = -1;
	char buf[20];
	UiEvent uie;

	while (1)
	{
		UiMenu uim(d);
		UiTextInput *t;

		bool b3, b35, b525, bs, bd;
		b3 = b35 = b525 = bs = bd = false;

		if (m_driveType[unit] == FD_30)  b3   = true;
		if (m_driveType[unit] == FD_35)  b35  = true;
		if (m_driveType[unit] == FD_525) b525 = true;
		if (m_driveHeads[unit] == 1) bs = true;		
		if (m_driveHeads[unit] == 2) bd = true;		

		uim.add(new UiSetting(SDLK_3,   b3,   "  3\" drive  ", d));
		uim.add(new UiSetting(SDLK_PERIOD, b35,  "  3.5\" drive  ", d));
		uim.add(new UiSetting(SDLK_5,   b525, "  5.25\" drive  ", d));
		uim.add(new UiSeparator(d));
		uim.add(t = new UiTextInput(SDLK_c, "  Number of cylinders: ___ ", d));
		uim.add(new UiSetting(SDLK_s,   bs, "  Single sided  ", d));
		uim.add(new UiSetting(SDLK_d,   bd, "  Double sided  ", d));
		uim.add(new UiSeparator(d));
		uim.add(new UiCommand(SDLK_ESCAPE, "  EXIT  ", d));

		sprintf(buf, "%d", m_driveCyls[unit]);
		t->setText(buf);	

		uim.setSelected(sel);
		d->centreContainer(uim);
		uie = uim.eventLoop();
		if (uie == UIE_QUIT) return uie;
		sel = uim.getSelected();
		switch(uim.getKey(sel))
		{
			case SDLK_3:   m_driveType[unit] = FD_30; break;
			case SDLK_PERIOD: m_driveType[unit] = FD_35; break;
			case SDLK_5:   m_driveType[unit] = FD_525; break;
			case SDLK_s:   m_driveHeads[unit] = 1; break;
			case SDLK_d:   m_driveHeads[unit] = 2; break;
			case SDLK_c:   m_driveCyls[unit] = atoi(t->getText().c_str()); break;
			case SDLK_ESCAPE: return UIE_CONTINUE;
			default: break;
		}	

	}
	return UIE_CONTINUE;
}


UiEvent JoyceFDC::driveSameAs(int unit, UiDrawer *d)
{
	char buf[40];
	UiMenu uim(d);	
	UiEvent e;
	int sel;

	for (int n = 0; n < unit; n++)	
	{
		sprintf(buf, "  Unit %d  ", n);
		uim.add(new UiCommand((SDLKey)(SDLK_0 + n), buf, d));
	}
	sel = m_driveMap[unit] % 4;
	if (sel >= unit) sel = -1;

	uim.setSelected(sel);
	d->centreContainer(uim);
	e = uim.eventLoop();
	if (e != UIE_OK) return e;
	m_driveMap[unit] = 12 + uim.getSelected();	
	return e;
}


JoyceModel JoyceFDC::adaptModel(JoyceModel j)
{
	bool is8256, is9256;

	is9256 = is8256 = false;
	if (m_driveMap[2] == 10 && m_driveMap[3] == 11) is9256 = true;
	if (m_driveMap[2] == 12 && m_driveMap[3] == 13) is8256 = true; 

	if (is8256 && j == PCW10   ) return PCW8000;
	if (is8256 && j == PCW9000P) return PCW9000;
	if (is9256 && j == PCW8000 ) return PCW10;
	if (is9256 && j == PCW9000 ) return PCW9000P;
	return j;
}	

void JoyceFDC::setModel(JoyceModel j)
{
	bool is8256, is9256;
	int n;

	JoyceDevice::setModel(j);

	is9256 = is8256 = false;
	if (m_driveMap[2] == 10 && m_driveMap[3] == 11) is9256 = true;
	if (m_driveMap[2] == 12 && m_driveMap[3] == 13) is8256 = true; 
	if (is8256 == 0 && is9256 == 0) return;	// Don't interfere with
						// custom setup.

	//
	// If there are changes, rewire drives 2 and 3 only.
	//
	if (is9256 && (j == PCW8000 || j == PCW9000))
	{
		m_driveMap[2] = 10; 
		m_driveMap[3] = 11;
        	for (n = 2; n < 4; n++) fd_eject(m_fdc.fdc_drive[n]);
		reset(2);
	}	
	if (is8256 && (j == PCW9000P || j == PCW10))
	{
		m_driveMap[2] = 12; 
		m_driveMap[3] = 13;
        	for (n = 2; n < 4; n++) fd_eject(m_fdc.fdc_drive[n]);
		reset(2);
	}	
		
}



UiEvent JoyceFDC::discsMenu(int x, int y, UiDrawer *d)
{
	UiEvent uie;
	int sel = -1;
	int nd;
	char *dtitle[4] = { "  Drive A (", "  Drive B (", 
			    "  Unit 2 (",  "  Unit 3 (" };

	while (1)
	{
		UiMenu uim(d);

		for (nd = 0; nd < 4; nd++) if (m_driveMap[nd] < 4)
		{
			string title = dtitle[nd];
			if (nd) uim.add(new UiSeparator(d));
			if (m_fdl[nd].fdl_filename[0]) 
			{
				title += displayName(m_fdl[nd].fdl_filename,40);
				title += ")  ";
				uim.add(new UiLabel(title, d));
				uim.add(new UiCommand((SDLKey)(SDLK_e + nd), "  Eject  ", d));
			}
			else
			{
				title += "Ejected)  ";
				uim.add(new UiLabel(title, d));
                         	uim.add(new UiCommand((SDLKey)(SDLK_i + nd), "  Insert...  ", d, UIG_SUBMENU));
			}
			uim.add(new UiSetting((SDLKey)(SDLK_r + nd), m_fdl[nd].fdl.fd_readonly, "  Read only  ", d));
		}
		uim.add(new UiSeparator(d));
		uim.add(new UiCommand(SDLK_m, "  Disc management...  ", d, UIG_SUBMENU));
		uim.add(new UiSeparator(d));
		uim.add(new UiCommand(SDLK_ESCAPE, "  EXIT  ", d));	
		uim.setX(x);
		uim.setY(y);
		uim.setSelected(sel);
		uie = uim.eventLoop();
		if (uie == UIE_QUIT) return uie;
		sel = uim.getSelected();
		switch(uim.getKey(sel))
		{
			case SDLK_e: fd_eject(&m_fdl[0].fdl); break; 
			case SDLK_f: fd_eject(&m_fdl[1].fdl); break; 
			case SDLK_g: fd_eject(&m_fdl[2].fdl); break; 
			case SDLK_h: fd_eject(&m_fdl[3].fdl); break; 
			case SDLK_i:
			case SDLK_j:
			case SDLK_k:
			case SDLK_l:
			{
				nd = uim.getKey(sel) - SDLK_i;
				string str = m_fdl[nd].fdl_filename;
				char *type = m_fdl[nd].fdl_type;
/* Old code for non-LIBDSK FDC
                                JoyceFileChooser f("  OK  ", d);
                                if (str != "") 
					f.initDir(str);
				else	f.initDir(findJoyceDir(FT_DISC, false));
                                uie = f.eventLoop();
                                if (uie == UIE_QUIT) return uie;
                                if (uie == UIE_OK) 
				{
					str = f.getChoice();
					fd_eject(&m_fdl[nd].fdl); 
					strcpy(m_fdl[nd].fdl_filename, str.c_str());	
				}
*/
				if (str == "") str = findJoyceDir(FT_DISC, false);
				uie = requestDsk(d, str, &type);
				if (uie == UIE_OK)
				{
					fd_eject(&m_fdl[nd].fdl); 
					strcpy(m_fdl[nd].fdl_filename, str.c_str());
					m_fdl[nd].fdl_type = type;	
				}				

			}
			break;
			case SDLK_m: uie = discManager(m_sys, d);
				     if (uie == UIE_QUIT) return uie;
				     break;
			case SDLK_r: m_fdl[0].fdl.fd_readonly = !m_fdl[0].fdl.fd_readonly; break;
			case SDLK_s: m_fdl[1].fdl.fd_readonly = !m_fdl[1].fdl.fd_readonly; break;
			case SDLK_t: m_fdl[2].fdl.fd_readonly = !m_fdl[2].fdl.fd_readonly; break;
			case SDLK_u: m_fdl[3].fdl.fd_readonly = !m_fdl[3].fdl.fd_readonly; break;


			case SDLK_ESCAPE: return UIE_OK;
			default: break;
		}	
	}

	return UIE_CONTINUE;
}




//
// When creating (or overriding) disc image file format, ask here.
// Note that there is a special version of this in JoyceSetup.cxx with
// special-case code to read in MD3 boot discs.
//
UiEvent JoyceFDC::getDskType(UiDrawer *d, int &fmt, bool create)
{
	UiEvent uie;
	int inMenu = 1;
	int sel = -1;
	int ofmt = fmt;

	while(inMenu)
	{
		UiMenu uim(d);

		uim.add(new UiLabel  ("      Drive type      ", d));
		uim.add(new UiSeparator(d));
		if (!create)
		{
			uim.add(new UiSetting(SDLK_0, (fmt == 0), "  Auto detect filetype  ", d));
		}
		uim.add(new UiSetting(SDLK_1, (fmt == 1), "  Floppy drive  ", d));
		uim.add(new UiSetting(SDLK_2, (fmt == 2), "  CPCEMU/JOYCE .DSK  ", d));
		uim.add(new UiSetting(SDLK_3, (fmt == 3), "  Raw disc image file  ", d));
		uim.add(new UiSetting(SDLK_4, (fmt == 4), "  MYZ80 hard drive  ", d));
		uim.add(new UiSeparator(d));
		uim.add(new UiCommand(SDLK_o, "  OK  ", d));
		uim.add(new UiCommand(SDLK_ESCAPE, "  Cancel  ", d));
		uim.setSelected(sel);
                d->centreContainer(uim);
		uie = uim.eventLoop();
		if (uie == UIE_QUIT || uie == UIE_CANCEL) 
		{
			fmt = ofmt;
			return uie;
		}
		sel = uim.getSelected();
		switch(uim.getKey(sel))
		{
			case SDLK_0: fmt = 0; break;
			case SDLK_1: fmt = 1; break;
			case SDLK_2: fmt = 2; break;
			case SDLK_3: fmt = 3; break;
			case SDLK_4: fmt = 4; break;
			case SDLK_ESCAPE: 
					fmt = ofmt;
					return UIE_CANCEL;
			case SDLK_o: inMenu = 0; break;
			default: break;
		}
	}
	return UIE_OK;
}



static char *dsktypes[] =
{
        NULL, "floppy", "dsk", "raw", "myz80"
};



// Ask for a DSK file to open / create
UiEvent JoyceFDC::requestDsk(UiDrawer *d, string &filename, char **type)
{
	int sel = -1;
	bool a,b,dsk;
	int fmt = 0;
	UiEvent uie;
	int inMenu = 1;
	
	while(inMenu)
	{
		UiMenu uim(d);

		if (filename == A_DRIVE) a = true; else a = false;
		if (filename == B_DRIVE) b = true; else b = false;
		if (!(a || b)) dsk = true; else dsk = false;

		uim.add(new UiLabel("  Enter filename for disc image  ", d));
		uim.add(new UiSeparator(d));
		uim.add(new UiLabel  ("  File/drive: " + displayName(filename, 40) + "  ", d));
		uim.add(new UiSetting(SDLK_a, a, "  Drive A: [" A_DRIVE "]  ",d));
		uim.add(new UiSetting(SDLK_b, b, "  Drive B: [" B_DRIVE "]  ",d));
		uim.add(new UiSetting(SDLK_d, dsk, "  Disc file...  ",d));
		uim.add(new UiSeparator(d));
		uim.add(new UiCommand(SDLK_v, "  Advanced...  ", d, UIG_SUBMENU));
		uim.add(new UiSeparator(d));
		uim.add(new UiCommand(SDLK_o, "  OK  ", d));
		uim.add(new UiCommand(SDLK_c, "  Cancel  ", d));
		uim.setSelected(sel);
                d->centreContainer(uim);
		uie = uim.eventLoop();
		if (uie == UIE_QUIT || uie == UIE_CANCEL) return uie;
		sel = uim.getSelected();
		switch(uim.getKey(sel))
		{
			case SDLK_a: filename = A_DRIVE; fmt = 1; break;
			case SDLK_b: filename = B_DRIVE; fmt = 1; break;
			case SDLK_c: return UIE_CANCEL;  break;
			case SDLK_o: inMenu = 0; break;
			case SDLK_d: 
                                {
                                    JoyceFileChooser f("  OK  ", d);
                                    f.initDir(filename);
                                    uie = f.eventLoop();
                                    if (uie == UIE_QUIT) return uie;
                                    if (uie == UIE_OK) 
				    {
					filename = f.getChoice();
					fmt = 2;	
				    }
                                }
                                break;
			case SDLK_v: 
				uie = getDskType(d, fmt, false);
                                if (uie == UIE_QUIT) return uie;
				break;
			default: break;
		}
	}

	// A filename has been entered. See if we can open the matching
	// DSK.
	DSK_PDRIVER drv;
	dsk_err_t err = dsk_open(&drv, filename.c_str(), dsktypes[fmt], NULL);
	if (!err)	// It can be opened. Great.
	{
		*type = dsktypes[fmt];
		dsk_close(&drv);
		return UIE_OK; 
	}	
	// File can't be opened. 
	
	UiMenu uim(d);
		
	string sl1;
	sl1 = "  Cannot open: ";
	sl1 += displayName(filename, 40);
	sl1 += "  ";

	uim.add(new UiLabel(sl1, d));
	uim.add(new UiLabel("", d));
	uim.add(new UiCommand(SDLK_c, "  Create it  ", d));
	uim.add(new UiCommand(SDLK_a, "  Abandon  ", d));
	d->centreContainer(uim);
	uie = uim.eventLoop();
	if (uie == UIE_QUIT || uie == UIE_CANCEL) return uie;
	if (uim.getKey(uim.getSelected()) != SDLK_c) return UIE_CANCEL;
	// We are at liberty to create it.

	if (!dsktypes[fmt])	// No type was specified
	{
		if (filename == A_DRIVE || filename == B_DRIVE) fmt = 1;
		else						fmt = 2;

		uie = getDskType(d, fmt, true);
		if (uie == UIE_QUIT || uie == UIE_CANCEL) return uie;
	}
	// Try to create a new blank disc image
	err = dsk_creat(&drv, filename.c_str(), dsktypes[fmt], NULL);
	if (!err)	// It can be created. Great.
	{
		*type = dsktypes[fmt];
		dsk_close(&drv);
		return UIE_OK; 
	}	
	UiMenu uim2(d);
		
	sl1 = "  Cannot open: ";
	sl1 += displayName(filename, 40);
	sl1 += "  ";

	uim2.add(new UiLabel(sl1, d));
	uim2.add(new UiCommand(SDLK_a, "  Abandon  ", d));
	d->centreContainer(uim2);
	uie = uim2.eventLoop();
	if (uie == UIE_QUIT || uie == UIE_CANCEL) return uie;

	return UIE_CANCEL;
}


