/************************************************************************

    JOYCE v1.90 - Amstrad PCW emulator

    Copyright (C) 1996, 2001  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/

#include "Joyce.hxx"
#include "UiContainer.hxx"
#include "UiLabel.hxx"
#include "UiSeparator.hxx"
#include "UiCommand.hxx"
#include "UiSetting.hxx"
#include "UiMenu.hxx"
#include "UiTextInput.hxx"
#include "UiScrollingMenu.hxx"
#include "JoyceFileList.hxx"
#include "JoyceFileChooser.hxx"

/* DPB for an 8Mb hard drive. Note that this doesn't match the geometry
 * used by LIBDSK, which has 1024k sectors. A little deblocking is called
 * for. 
 *
 * This DPB has 16k/track (32 512 byte sectors), and 512 tracks. LIBDSK uses
 *             128k/track (128 1k sectors) and 64 tracks.
 * */
static const byte dpb[] = { 0x80, 0x00, 0x05, 0x1F, 0x01, 0xFF, 0x07, 0xFF,
			    0x03, 0xFF, 0x00, 0x00, 0x80, 0x00, 0x00, 0x02,
			    0x03 };




JoyceHDC::JoyceHDC() : JoyceDevice("hdc")
{
	int n;
	char buf[20];
	
	m_allowFloppy = false;
	for (n = 0; n < 16; n++) 
	{
		sprintf(buf, "%c.dsk", n + 'a');
		m_drvName[n] = findJoyceFile(FT_DISC, buf, "rb"); 
		m_drvUnit[n] = NULL;
		m_drvType[n] = "myz80";
		deblockSector [n] = -1;
		deblockPending[n] = 0;
	}
}


JoyceHDC::~JoyceHDC()
{
	for (int n = 0; n < 16; n++) 
	{
		closeImage(n);
	}
}

void JoyceHDC::reset(void)
{
	for (int n = 0; n < 16; n++) 
	{
		closeImage(n);
	}
}


bool JoyceHDC::hasSettings(SDLKey *key, string &caption)
{
	*key = SDLK_e;
	caption = "  Extra drives  ";	
	return true;
}

UiEvent JoyceHDC::setupDrive(UiDrawer *d, int drive)
{
        int sel = -1;
        bool a,b,dsk, myz80, autoType;
        string filename = m_drvName[drive];
        UiEvent uie;
        int inMenu = 1;
	char prompt[50];

	sprintf(prompt, "  Options for drive %c:  ", drive + 'A');

        while(inMenu)
        {
                UiMenu uim(d);

                if (filename == A_DRIVE) a = true; else a = false;
                if (filename == B_DRIVE) b = true; else b = false;
                if (!(a || b)) dsk = true; else dsk = false;

		autoType = (m_drvType[drive] == NULL);
		if (!autoType) myz80 = !strcmp(m_drvType[drive], "myz80");
		else	       myz80 = false;
                uim.add(new UiLabel(prompt, d));
                uim.add(new UiSeparator(d));
                uim.add(new UiLabel  ("  File/drive: " + displayName(filename, 40) + "  ", d));
                uim.add(new UiSetting(SDLK_a, a, "  Drive A: [" A_DRIVE "]  ",d));
                uim.add(new UiSetting(SDLK_b, b, "  Drive B: [" B_DRIVE "]  ",d));
		uim.add(new UiSetting(SDLK_d, dsk, "  Disc file...  ",d));
		uim.add(new UiSeparator(d));
		uim.add(new UiSetting(SDLK_f, autoType, "  Floppy drive  ", d));
		uim.add(new UiSetting(SDLK_h, myz80,    "  Hard drive  ", d));
		uim.add(new UiSeparator(d));
		uim.add(new UiCommand(SDLK_ESCAPE, "  EXIT  ", d));
		uim.setSelected(sel);
		d->centreContainer(uim);
		uie = uim.eventLoop();
		sel = uim.getSelected();
		switch(uim.getKey(sel))
		{
			case SDLK_a: filename = A_DRIVE; m_drvType[drive] = NULL; break;
			case SDLK_b: filename = B_DRIVE; m_drvType[drive] = NULL; break;
			case SDLK_d:
                             {
                             JoyceFileChooser f("  OK  ", d);
                             f.initDir(filename);
                             uie = f.eventLoop();
                             if (uie == UIE_QUIT) return uie;
                             if (uie == UIE_OK) 
				{
				filename = f.getChoice();
				m_drvType[drive] = "myz80";
				}
                             }
                             break;
			case SDLK_f: m_drvType[drive] = NULL; break;
			case SDLK_h: m_drvType[drive] = "myz80"; break;
			case SDLK_ESCAPE:
        			m_drvName[drive] = filename;
				return UIE_OK;
			default: break;	
		}
	}
	return UIE_CONTINUE;
}


UiEvent JoyceHDC::settings(UiDrawer *d)
{
	UiEvent uie;
	int n;
	int sel = -1;
	while(1)
	{
		UiMenu uim(d);
		uim.add(new UiLabel("  Emulated extra drives  ", d));	
		uim.add(new UiSeparator(d));
		for (n = 2; n < 16; n++)
		{
			SDLKey k = (SDLKey)(SDLK_a + n);
			char buf[30];

			if (n == 12) continue;
			sprintf(buf, "  Settings for drive %c:  ", n + 'A');
			uim.add(new UiCommand(k, buf, d));
		}
		uim.add(new UiSeparator(d));
		uim.add(new UiCommand(SDLK_ESCAPE, "  EXIT  ", d));
                d->centreContainer(uim);
                uim.setSelected(sel);
                uie = uim.eventLoop();

                if (uie == UIE_QUIT) return uie;
                switch(uim.getKey(sel = uim.getSelected()))
                {
			case SDLK_ESCAPE:
				reset();
				return UIE_OK;
			default:
				n = uim.getKey(sel) - SDLK_a;
				uie = setupDrive(d, n);
                		if (uie == UIE_QUIT) return uie;
				break;
		}
	}
	return UIE_CONTINUE;
}


bool JoyceHDC::parseDriveNode(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
	char *s;
	int id;

	s = (char *)xmlGetProp(cur, (xmlChar *)"letter");
	if (!s || s[1] != 0 || s[0] < 'a' || s[0] > 'p')
        {
                joyce_dprintf("<drive> must have a letter= field\n");
                return true;
        }
	id = s[0] - 'a';
	free(s);
	parseFilename(doc, ns, cur, "filename", m_drvName[id]);
	s = (char *)xmlGetProp(cur, (xmlChar *)"type");
	if (s)
	{
		if (!strcmp(s, "auto")) m_drvType[id] = NULL;
		else
		{
			m_userType[id] = s;
			m_drvType[id]  = m_userType[id].c_str();
		}
		free(s);
	}
	return true;	
}



bool JoyceHDC::parseNode(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
        // Settings for individual drives
        cur = cur->xmlChildrenNode;
        while (cur != NULL)
        {
                if (!strcmp((char *)cur->name, "drive") && (cur->ns == ns))
                {
                        if (!parseDriveNode(doc, ns, cur)) return false;
                }
                cur = cur->next;
        }
	return true;
}

bool JoyceHDC::storeDriveNode(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur, int n)
{
	char buf[3];

	sprintf(buf, "%c", n + 'a');
	xmlSetProp(cur, (xmlChar *)"letter", (xmlChar *)buf);
	storeFilename(doc, ns, cur, "filename", m_drvName[n]);
	if (m_drvType[n]) xmlSetProp(cur, (xmlChar *)"type", (xmlChar *)m_drvType[n]);
	else		  xmlSetProp(cur, (xmlChar *)"type", (xmlChar *)"auto");
	return true;
}


bool JoyceHDC::storeNode(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
        bool some;
        char buf[20];
        xmlNodePtr drv;

        for (int n = 0; n < 16; n++)
        {
                some = false;
                sprintf(buf, "%c", n + 'a');
                for (drv = cur->xmlChildrenNode; drv; drv = drv->next)
                {
                        if (strcmp((char *)drv->name, "drive")) continue;

                        char *s = (char *)xmlGetProp(drv, (xmlChar *)"letter");
                        if (!s) continue;
                        if (strcmp(s, buf))
                        {
                                free(s);
                                continue;
                        }
                        some = true;
                        storeDriveNode(doc, ns, drv, n);
		}
                // No existing <drive> node; create one.
                if (!some)
                {
                        drv = xmlNewNode(ns, (xmlChar *)"drive");
                        xmlSetProp(drv, (xmlChar *)"letter", (xmlChar *)buf);
                        xmlAddChild(cur, drv);
                        storeDriveNode(doc, ns, drv, n);
                }
        }
        return true;
}




void JoyceHDC::fidPuts(word addr, const string s)
{
	for (unsigned n = 0; n < s.length(); n++)
	{
		::store(addr++, s[n]);
	}
	::store(addr++, 0xFF);
}


int JoyceHDC::openImage(int img)
{
	string dname;
	dsk_err_t err;

	if (m_drvUnit[img]) return 1;

	dname = findJoyceFile(FT_DISC, m_drvName[img], "r+b");
	// Try to open existing file. If that fails, create a new one.
	err = dsk_open(&m_drvUnit[img], m_drvName[img].c_str(), m_drvType[img], NULL);
	if (err) err = dsk_creat(&m_drvUnit[img], m_drvName[img].c_str(), m_drvType[img], NULL);
	if (!err) err = dsk_getgeom(m_drvUnit[img], &m_drvDg[img]);
	if (err)
	{
		joyce_dprintf("Failed to open %s: %s\n", 
				m_drvName[img].c_str(), dsk_strerror(err));
		return 0;
	}
	return 1;		/* File opened */
}

int JoyceHDC::closeImage(int img)
{
	if (m_drvUnit[img])
	{
		if (deblockPending[img]) deblockWrite(img, deblockSector[img]);

 		dsk_close(&m_drvUnit[img]);
	}
	return 1;
}


bool JoyceHDC::deblockRead (int img, long sector)
{
	dsk_err_t err;

	if (sector == deblockSector[img]) return true;

	if (deblockPending[img]) deblockWrite(img, deblockSector[img]);
	if ((err = dsk_lread(m_drvUnit[img], &m_drvDg[img], deblock[img], sector)))
	{
		m_lastErr = dsk_strerror(err);	
		return false;
	}
	deblockSector[img] = sector;
	return true;
}

bool JoyceHDC::deblockWrite(int img, long sector)
{
	dsk_err_t err;

	if ((err = dsk_lwrite(m_drvUnit[img], &m_drvDg[img], deblock[img], sector)))
	{
		m_lastErr = dsk_strerror(err);	
		return false;
	}
	deblockPending[img] = false;
	return true;
}


void JoyceHDC::ret_err(Z80 *R)
{
	R->a = 1;
	R->f &= ~(FLG_C | FLG_H);	/* No Carry, No Halfcarry, Zero */
	R->f |= FLG_Z;
}

void JoyceHDC::ret_ok(Z80 *R)
{
	m_lastErr = "";
	R->a = 0;
	R->f |= FLG_C;		/* Carry set, no Halfcarry */
	R->f &= ~FLG_H;
}


void JoyceHDC::fidLogin(Z80 *R)
{
	int nx;
	byte drv = R->b;
	dsk_err_t err;
	
	m_lastDrv = drv;
	if (!openImage(drv)) 
	{ 
		m_lastErr = "Could not open drive";
		ret_err(R); 
		return; 
	}

	/* MYZ80 drives have a fixed geometry */
	if (m_drvType[drv] && !strcmp(m_drvType[drv], "myz80"))
	{
		for (nx = 0; nx < 17; nx++) ::store(R->ix + nx, dpb[nx]);
		ret_ok(R);
		return;		
	}	
	/* XXX What about CPC discs? */
	/* Load the boot sector into deblock[drv] */
	if ((err = dsk_lread(m_drvUnit[drv], &m_drvDg[drv], deblock[drv], 0)))
	{
		m_lastErr = dsk_strerror(err);	
		ret_err(R);
		return;	
	}
        /* v1.31: Support PCW16 extended formats */
        if (deblock[drv][0] == 0xEB || deblock[drv][0] == 0xE9)
        {
                if (!memcmp(deblock[drv] + 0x2B, "CP/M", 4) &&
                    !memcmp(deblock[drv] + 0x33, "DSK",  3) &&
                    !memcmp(deblock[drv] + 0x7C, "CP/M", 4))
                {
                        memcpy(deblock[drv] + 0, deblock[drv] + 0x80, 16);
                }
        }       /* end v1.31 change */
        makeDpb(R->ix, deblock[drv], 0);
	ret_ok(R);
}

void JoyceHDC::fidRead(Z80 *R)	
{
	byte *data;
	byte img = R->b;
	unsigned trk = R->getHL();
	unsigned sec = R->getDE();
	m_lastDrv = img;
	long lsect, dlsect;
	word spt;
	dsk_err_t err;

	if (!openImage(img)) 
	{ 
		m_lastErr = "Could not open drive";
		ret_err(R); 
		return; 
	}
	
	spt = fetch2(R->ix);			// Sectors/track [DPB]

	// lsect = address of sector (in CP/M's 512-byte units) 
	lsect = ((spt/4) * trk) + sec;
	// dlsect = address of sector in the drive's native sectors
	dlsect = (lsect * 512) / m_drvDg[img].dg_secsize;
	// Physical sector is smaller. Combine blocks.
	if (m_drvDg[img].dg_secsize < 512)
	{
		data = deblock[img];
		for (int ns = 0; ns < 512; ns += m_drvDg[img].dg_secsize)
		{
			err = dsk_lread(m_drvUnit[img], &m_drvDg[img], 
				data + ns, dlsect);
			if (err)
			{
				m_lastErr = dsk_strerror(err);
				ret_err(R);
				return;
			}
			++dlsect;
		}
	}	
	// Sector sizes match
	if (m_drvDg[img].dg_secsize == 512)
	{
		err = dsk_lread(m_drvUnit[img], &m_drvDg[img], deblock[img], lsect);
		if (err)
		{
			m_lastErr = dsk_strerror(err);
			ret_err(R);
			return;
		}
		data = deblock[img];
	}
	// Physical sector size is bigger. Deblock.
	if (m_drvDg[img].dg_secsize > 512)
	{
		if (!deblockRead(img, dlsect)) { ret_err(R); return; }
		if (lsect & 1) data = deblock[img] + 512;
		else	       data = deblock[img];
	}

	for (int nx = 0; nx < 512; nx++) 
	{
		::store(R->iy++, data[nx]);
	}
	ret_ok(R);
}


void JoyceHDC::fidWrite(Z80 *R)	
{
	byte *data;
	byte img = R->b;
	unsigned trk = R->getHL();
	unsigned sec = R->getDE();
	m_lastDrv = img;
	long lsect, dlsect;
	word spt;
	dsk_err_t err;

	if (!openImage(img)) 
	{ 
		m_lastErr = "Could not open drive";
		ret_err(R); 
		return; 
	}
	
	spt = fetch2(R->ix);			// Sectors/track [DPB]

	// lsect = address of sector (in CP/M's 512-byte units) 
	lsect = ((spt/4) * trk) + sec;
	// dlsect = address of sector in the drive's native sectors
	dlsect = (lsect * 512) / m_drvDg[img].dg_secsize;

	// Copy the data from Z80 memory into the buffer.
	data = deblock[img];
	// If we have to deblock, make sure the buffer has the correct
	// record in it.
	if (m_drvDg[img].dg_secsize > 512)
	{
		if (!deblockRead(img, dlsect)) { ret_err(R); return; }
 		if (lsect & 1) data += 512;
	}
	for (int nx = 0; nx < 512; nx++) 
	{
		data[nx] = fetch(R->iy++);
	}

	// Physical sector is smaller. Write multiple sectors.
	if (m_drvDg[img].dg_secsize < 512)
	{
		data = deblock[img];
		for (int ns = 0; ns < 512; ns += m_drvDg[img].dg_secsize)
		{
			err = dsk_lwrite(m_drvUnit[img], &m_drvDg[img], 
				data + ns, dlsect);
			if (err)
			{
				m_lastErr = dsk_strerror(err);
				ret_err(R);
				return;
			}
			++dlsect;
		}
	}	
	// Sector sizes match
	if (m_drvDg[img].dg_secsize == 512)
	{
		err = dsk_lwrite(m_drvUnit[img], &m_drvDg[img], deblock[img], lsect);
		if (err)
		{
			m_lastErr = dsk_strerror(err);
			ret_err(R);
			return;
		}
		data = deblock[img];
	}
	// Physical sector size is bigger. Deblock.
	if (m_drvDg[img].dg_secsize > 512)
	{
		deblockPending[img] = true;
		deblockSector[img]  = dlsect;
		if (R->c == 1)	// Immediate write
			if (!deblockWrite(img, dlsect)) { ret_err(R); return; }
	}
	ret_ok(R);
}


void JoyceHDC::fidFlush(Z80 *R)	
{
	int img = R->b;
	closeImage(img);
	ret_ok(R);	
}

void JoyceHDC::fidMessage(Z80 *R)	/* (v1.22) expand disc error message */
{
	char msg[94];

	sprintf(msg,"%c: %-92s", m_lastDrv + 'A', m_lastErr.c_str());

	R->setHL(m_fidText);
	R->f |= FLG_C;
	fidPuts(m_fidText, msg);			
}


void JoyceHDC::setupPars(Z80 *R)		/* v1.22: Determine whether a drive is floppy */
{				/* or hard, and set HL IX IY accordingly */
	int img = R->b;
	if (!m_drvType[img] || strcmp(m_drvType[img], "myz80"))	// Floppy 
	{
		R->setHL(91);
		R->ix = 64;
		R->iy = 1024;
	}
	else
	{
		R->setHL(513);
		R->ix = 0;
		R->iy = 0;
	}
}



void JoyceHDC::fidEms(Z80 *R)
{
	static int nhooked;
	static byte cdrv;
	char str[90];

/* This is called with: 

	A = 7
	B = flags, currently 0
	C = drive letter, currently 0FFh
	D = error:  0 => a drive has been hooked, E is its letter
		    1 => hook failure, no more memory
		    2 => hook failure, drive letter in use
		    3 => version control error
		    4 => start of fid_ems routine
	HL -> 93-byte message buffer

    returns: A=0 => ret NC
	     A=1 => ret C
	     A=2 => attempt to hook another disc drive
		      B=new drive letter

*/

	switch (R->d)
	{
		case 4:			/* Start of FIDEMS */
		if (R->c != 0xFF) m_allowFloppy = true;
		hooked = "";	/* No drives hooked yet */
		nhooked = 0;
		cdrv = 2;	/* Start trying to hook at C: */
		R->b = cdrv;
		R->a = 2;
		m_fidText = R->getHL();	/* (v1.22) save text buffer */
		setupPars(R);
		return;
	
		case 3:			/* Version control */
		fidPuts(R->getHL(), "JOYCE driver: Incorrect FID environment\r\n");
		R->a = 0;
		return;

		case 2:			/* Drive letter in use */
		if (cdrv < 0x0f)	/* (v1.22) Failed to get C: - O: */
		{			/* This is because that */
					/* letter's in use, so try the */
			R->a = 2;	/* next letter */
			++cdrv;
			R->b = cdrv;
			setupPars(R);
		}
		else if (nhooked)
		{
			sprintf(str,"JOYCE driver on drives %s\r\n", hooked.c_str());
			fidPuts(R->getHL(),str);
			R->a = 1;
		}
		else
		{
			fidPuts(R->getHL(), "JOYCE driver: No free drive letters\r\n");
			R->a = 0;
		}
		return;
		case 1:			/* No free memory */
		if (nhooked)
		{
			sprintf(str,"JOYCE driver on drives %s\r\n", hooked.c_str());
			fidPuts(R->getHL(),str);
			R->a = 1;
			return;
		}
		else
		{
			fidPuts(R->getHL(), "JOYCE driver: No memory available\r\n");
			R->a = 0;
			return;
		}
		case 0:			/* Drive hooked OK */
		nhooked++;
		sprintf(str,"%c: ", R->e + 'A'); 
		hooked += str;
		R->a = 2;
		++cdrv;
		R->b = cdrv;
		setupPars(R);
		return;
	}

}





static word to_al0(word blocks)
{
	word al0 = 0;

	while (blocks--) al0 = (al0 >> 1) | 0x8000;
	return al0;
}


static byte tomask(byte b)
{
	byte mask = 0;

	while (b--) mask = (mask << 1) | 1;
	return mask;
}


static word sizefrom(byte b)
{
	word W = 128;
	
	while (b--) W *= 2;

	return W;
}


void JoyceHDC::makeDpb(word dpb, byte *format, int is_xdpb)
{
	word BLOCK_SIZE, OFF, NUM_TRACKS, NUM_SECTORS, DIR_BLOCKS,
	     SEC_SIZE;

	word SPT, BSH, BLM, DSM, DRM, AL0, AL1, EXM, CKS, PSH, PHM;

	BLOCK_SIZE  = sizefrom(format[6]);
	OFF         = format[5];
	NUM_TRACKS  = format[2]; if (format[1] & 3) NUM_TRACKS *= 2;
	NUM_SECTORS = format[3];
	DIR_BLOCKS  = format[7];
	SEC_SIZE    = sizefrom(format[4]);

	SPT = NUM_SECTORS * SEC_SIZE  / 128; 	
	BSH = format[6];	
	BLM = tomask(format[6]);
	DSM = ((NUM_TRACKS - OFF) * NUM_SECTORS * SEC_SIZE / BLOCK_SIZE) - 1;
	DRM = (DIR_BLOCKS * BLOCK_SIZE / 32) - 1;
	AL0 = (to_al0(DIR_BLOCKS) >> 8);
	AL1 = (to_al0(DIR_BLOCKS) & 0xFF);
	if (DSM < 256) EXM = (BLOCK_SIZE >> 10) - 1;
	else	       EXM = (BLOCK_SIZE >> 11) - 1;
	CKS = (DRM + 1) / 4;
	PSH = format[4];
	PHM = tomask(format[4]);

	::store2(dpb,     SPT);
	::store (dpb + 2, BSH);
	::store (dpb + 3, BLM);
	::store (dpb + 4, EXM);
	::store2(dpb + 5, DSM);
	::store2(dpb + 7, DRM);
	::store (dpb + 9, AL0);
	::store (dpb +10, AL1);
	::store2(dpb +11, CKS);
	::store2(dpb +13, OFF);
	::store (dpb +15, PSH);
	::store (dpb +16, PHM);
}



