/************************************************************************

    JOYCE v1.90 - Amstrad PCW emulator

    Copyright (C) 1996, 2001  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/

#include "Joyce.hxx"
#include "Joycevga.hxx"
#include "inline.hxx"
#include "UiContainer.hxx"
#include "UiMenuBar.hxx"
#include "UiLabel.hxx"
#include "UiCommand.hxx"
#include "UiSetting.hxx"
#include "UiMenu.hxx"
#include "UiSeparator.hxx"
#include "UiMenuBarCommand.hxx"

JoyceMenuTerm::JoyceMenuTerm(JoyceSystem *s) : JoyceSdlTerm("menu", s)
{
	XLOG("JoyceMenuTerm::JoyceMenuTerm()");
	m_menuMutex = false;

        m_clrBG.r   = 0x00; m_clrBG.g   = 0x00; m_clrBG.b   = 0xAA;
        m_clrFG.r   = 0x55; m_clrFG.g   = 0xFF; m_clrFG.b   = 0xFF;

	m_popupBacking = NULL;

	m_menuF9 = new UiMenuBar(this);
	m_menuF9->add(new UiMenuBarCommand(SDLK_ESCAPE, "ESC=Return to JOYCE", this));
	m_menuF9->add(new UiMenuBarCommand(SDLK_F1,     "f1=Help", this));
	m_menuF9->add(new UiMenuBarCommand(SDLK_F2,     "f2=Disc", this));
	m_menuF9->add(new UiMenuBarCommand(SDLK_F4,     "f4=Printer", this));
	m_menuF9->add(new UiMenuBarCommand(SDLK_F5,     "f5=Reboot", this));
	m_menuF9->add(new UiMenuBarCommand(SDLK_F6,     "f6=Settings", this));
	m_menuF9->add(new UiMenuBarCommand(SDLK_F7,     "f7=Debug", this));
	m_menuF9->add(new UiMenuBarCommand(SDLK_F10,    "f10=Leave JOYCE", this));
}


JoyceMenuTerm::~JoyceMenuTerm()
{
	delete m_menuF9;
}


void JoyceMenuTerm::setSysVideo(JoyceSdlContext *c)
{
	if (m_term) delete m_term;
	m_term = NULL;
	if (c) m_term = new SdlTerm(c->getSurface());
	if (m_term)
	{
		m_term->setForeground(m_clrFG);
		m_term->setBackground(m_clrBG);
	}
	JoyceSdlTerm::setSysVideo(c);
}

void JoyceMenuTerm::onGainFocus()
{
	if (m_term)
	{
		m_term->setForeground(m_clrFG);
		m_term->setBackground(m_clrBG);
		m_term->enableCursor(false);
	}	
	JoyceSdlTerm::onGainFocus();
}

/* Ask the user if they want to quit. Returns only if they don't */

UiEvent JoyceMenuTerm::confirm(int x, int y, const string prompt)
{	
	int lpad, rpad, tlen;
	string title;
	UiEvent e;
	UiMenu mc(this);

	if (prompt != "") 
	{
		tlen = prompt.size();
		if (tlen >= 14) title = prompt.substr(0, 15);
		else
		{
			title = "";
			lpad = (15 - tlen) / 2;
			rpad = 15 - (tlen + lpad);
			while (lpad--) title += " ";
			title += prompt;
			while (rpad--) title += " ";	
		}
                mc.add(new UiLabel(title, this));
	}
	mc.add(new UiLabel(" Are you sure? ", this))
	  .add(new UiSeparator(this))
	  .add(new UiCommand(SDLK_n, "  No",  this))
	  .add(new UiCommand(SDLK_y, "  Yes", this));
	mc.setX(x);
	mc.setY(y);

	e = mc.eventLoop();

	if (e == UIE_QUIT) return e;

	if (mc.getKey(mc.getSelected()) == SDLK_y) return UIE_OK;
	return UIE_CANCEL;
}





UiEvent JoyceMenuTerm::report(const string s, const string prompt)
{
	UiMenu uim(this);
	int x,y,mw;

	x = s.size() + 2;
	y = prompt.size() + 2;

	mw = (x >= y) ? x : y;	

	uim.add(new UiLabel(s, this))
	   .add(new UiSeparator(this))
	   .add(new UiCommand(SDLK_o, prompt, this));
        y = (ui_screen_h() -  6 *       ui_char_h()) / 2;
        x = (ui_screen_w() - (2 + mw) * ui_char_w()) / 2;

	x -= (x % ui_char_w());	
	y -= (y % ui_char_h());

	uim.setX(x);
	uim.setY(y);
	return uim.eventLoop();
}
	


void JoyceMenuTerm::exitMenu(void)
{
	int x, y;
	UiEvent uie;

	if (m_menuMutex) return;

	m_menuMutex = true;
        m_oldTerm = m_sys->m_activeTerminal;
	SDL_BlitSurface(m_sysVideo->getSurface(), NULL,
			m_framebuf->getSurface(), NULL);
        m_sys->selectTerminal(this);
	SDL_ShowCursor(1);

	y = (ui_screen_h() -  6 * ui_char_h()) / 2;
	x = (ui_screen_w() - 17 * ui_char_w()) / 2;

	y -= (y % ui_char_h());
	x -= (x % ui_char_w());
	
	uie = confirm(x, y, "Leave JOYCE");
	if (uie == UIE_OK || uie == UIE_QUIT) goodbye(99);

	m_sys->selectTerminal(m_oldTerm);
	m_menuMutex = 0;
}

/* For dumping the CPU state */

static void put_flags(char *s, byte F)
{
	strcat(s, F & 0x80 ? "S" : "s");
	strcat(s, F & 0x40 ? "Z" : "z");
	strcat(s, F & 0x10 ? "H" : "h");
	strcat(s, F & 0x04 ? "P" : "p");
	strcat(s, F & 0x02 ? "N" : "n");
	strcat(s, F & 0x01 ? "C" : "c");
}



UiEvent JoyceMenuTerm::cpuState(Z80 *R)
{
	static char s[5][50];
	int x, y, n;
	word t[3];
	UiMenu uim(this);

	x = (ui_screen_w() - (ui_char_w() * 52)) / 2;
	y = (ui_screen_h() - (ui_char_h() * 10)) / 2;

	x -= (x % ui_char_w());
	y -= (y % ui_char_h());

	t[0] = fetch2(R->sp  );
	t[1] = fetch2(R->sp+2);
	t[2] = fetch2(R->sp+4);

	sprintf(s[0],"A=%2.2x BC=%4.4x DE=%4.4x HL=%4.4x  ",
			R->a, R->getBC(), R->getDE(), R->getHL());
	put_flags(s[0], R->f);

        sprintf(s[1],"A'%2.2x BC'%4.4x DE'%4.4x HL'%4.4x  ",
                        R->a1, R->getBC1(), R->getDE1(), R->getHL1());
	put_flags(s[1], R->f1);

	sprintf(s[2],"IX=%4.4x IY=%4.4x PC=%4.4x SP=%4.4x I=%2.2x IFF2=%2.2x",
		  R->ix, R->iy, R->pc, R->sp, R->i, R->iff2);

	sprintf(s[3], "Stack: %4.4x %4.4x %4.4x", t[0], t[1], t[2]);
	sprintf(s[4]," %4.4x: %2.2x %2.2x %2.2x %2.2x <> %2.2x %2.2x "
                  "%2.2x %2.2x %2.2x %2.2x %2.2x", R->pc - 4,
                  fetch(R->pc-4), 
                  fetch(R->pc-3), fetch(R->pc-2), fetch(R->pc-1), 
                  fetch(R->pc), fetch(R->pc+1), fetch(R->pc+2),
		  fetch(R->pc+3), fetch(R->pc+4), fetch(R->pc+5), 
                  fetch(R->pc+6) );

	uim.add(new UiLabel("                  Z80 CPU State                  ", this))
	   .add(new UiSeparator(this));
	
        for (n = 0; n < 5; n++) 
	{
		uim.add(new UiLabel(s[n], this));
	}
	uim.setX(x);
	uim.setY(y);
	uim.add(new UiSeparator(this))
	   .add(new UiCommand  (SDLK_ESCAPE,  "  OK  ", this));
	if (uim.eventLoop() == UIE_QUIT) return UIE_QUIT;
	return UIE_OK;
}


UiEvent JoyceMenuTerm::menuF1(int x, int y)
{
	UiMenu uim(this);
	UiEvent uie;

        x = x - (x % ui_char_w());
        y = y - (y % ui_char_h());

	uim.add(new UiCommand (SDLK_a, "  About JOYCE...   ", this, UIG_SUBMENU));
	uim.add(new UiCommand (SDLK_k, "  Keyboard layout  ", this));
	uim.setX(x);
	uim.setY(y);

	uie = uim.eventLoop();

	if (uie < 0) return uie;
	switch(uim.getKey(uim.getSelected()))
	{
		case SDLK_a: return aboutBox();
		case SDLK_k: 
			drawKeyboard();
		default:
			break;
	}
	return UIE_CONTINUE;
}


UiEvent JoyceMenuTerm::debugMenu(Z80 *R, int x, int y)
{
	UiMenu uim(this);
	UiEvent uie;

	x = x - (x % ui_char_w());
	y = y - (y % ui_char_h());

	uim.add(new UiSetting(SDLK_d, false, "  Debugger  ",  this))
	   .add(new UiCommand(SDLK_c, "  CPU state  ", this, UIG_SUBMENU))
	   .add(new UiCommand(SDLK_r, "  RAM dump   ", this))
	   .add(new UiSeparator(this))
	   .add(new UiCommand(SDLK_ESCAPE, "  EXIT  ", this));
	uim.setX(x);
	uim.setY(y);

	while (1)
	{
		((UiSetting &)uim[0]).setChecked(m_sys->getDebug());
		uie = uim.eventLoop();

		if (uie < 0) return uie;
		switch(uim.getKey(uim.getSelected()))
		{
			case SDLK_d:	m_sys->setDebug(!m_sys->getDebug());
					return UIE_OK;
			
			case SDLK_c:	uie= cpuState(R); 
					if (uie== UIE_QUIT) return uie;
					break;	

			case SDLK_r:
					{
						FILE *fp = fopen("joyce.ram", "wb");
						fwrite(PCWRAM, 16384, 10, fp);
						fclose(fp);
					}	
					uie = report("Bottom 160k dumped as joyce.ram", "OK");
					if (uie == UIE_QUIT) return uie;
					break;
			default:	return UIE_CONTINUE;
		}
	}
	return UIE_CONTINUE;
}

#define MENU_X(ww)  (*m_menuF9)[m_menuF9->getSelected()].getX() \
		 + ((*m_menuF9)[m_menuF9->getSelected()].getWidth() / 2) \
                            - (ww * ui_char_w() / 2)



void JoyceMenuTerm::mainMenu(Z80 *R)	/* Handle the menu */
{
	UiEvent uie;
	int y, x;
	int inMenu;

	if (m_menuMutex) return;	/* Prevent recursion */

	m_menuMutex=1;
        m_oldTerm = m_sys->m_activeTerminal;
        SDL_BlitSurface(m_sysVideo->getSurface(), NULL,
                        m_framebuf->getSurface(), NULL);
        m_sys->selectTerminal(this);
	menuDraw();
        SDL_ShowCursor(1);
	
	do
	{
		inMenu = 1;
		uie = m_menuF9->eventLoop();

		if (uie == UIE_QUIT) goodbye(99); /* SDL_QUIT */
		
		y = m_menuF9->getY() + m_menuF9->getHeight();

		switch(m_menuF9->getKey(m_menuF9->getSelected()))
		{
			case SDLK_F7:	/* D = Debug mode */
			uie = debugMenu(R, MENU_X(15), y);
			if (uie == UIE_OK) inMenu = 0; 
			else if (uie == UIE_QUIT) goodbye(99);
			else menuDraw();
			break;
			
			case SDLK_ESCAPE:	/* ESC */
       			inMenu = 0;
			break;

			case SDLK_F5:	/* Reboot */
			uie = confirm(MENU_X(17), y, "");

			if (uie == UIE_QUIT) goodbye(99);
			if (uie == UIE_OK) 
			{
				m_sys->setResetFlag(); 
				inMenu = 0; 
			}
			menuDraw();
			break;

			case SDLK_F4:	/* Printer */
			if (m_sys->getModel() == PCW9000 || 
                            m_sys->getModel() == PCW9000P)
			     uie = m_sys->m_daisy.control(this);	
			else uie = m_sys->m_matrix.control(this);
			if (uie == UIE_QUIT) goodbye(99);
			menuDraw();
			break;

			case SDLK_F2:	/* Disc */
			x = MENU_X(20);
			x -= (x % ui_char_w());
			uie = m_sys->m_fdc.discsMenu(x, y - (y % ui_char_h()), this);
			if (uie == UIE_QUIT) goodbye(99);
			menuDraw();
			break;

			case SDLK_F1:	/* Help */
			uie = menuF1(MENU_X(19), y);
			if (uie == UIE_QUIT) goodbye(99);
			break;

			case SDLK_F6:	/* Settings */
			uie = m_sys->settings(this, MENU_X(19), y);
			if (uie == UIE_QUIT) goodbye(99);
			menuDraw();
			break;		

			case SDLK_F10:	/* f10 = Exit */
			x = ui_screen_w() - 17 * ui_char_w();
			uie = confirm(x, y, "");
			if (uie == UIE_OK || uie == UIE_QUIT) goodbye(99);
			menuDraw();
			break;

			default: uie = UIE_CONTINUE; break;
		}	/* switch */
		if (uie == UIE_QUIT) goodbye(99);	// Quit event 
	} while(inMenu);

	// Cleanup
	m_sys->selectTerminal(m_oldTerm);
        m_menuMutex=0;

}



SDL_Surface *JoyceMenuTerm::getSurface()
{
	return m_term->getSurface();
}


SDL_Surface *JoyceMenuTerm::saveSurface(SDL_Surface *s, SDL_Rect &rc)
{
        SDL_Rect rcto;
        SDL_Surface *s2;

        rcto.x = rcto.y = 0;
        rcto.w = rc.w;
        rcto.h = rc.h;

        s2 = SDL_AllocSurface(SDL_SWSURFACE, rcto.w, rcto.h,
				s->format->BitsPerPixel, 0, 0, 0, 0);

        if (!s2) return NULL;
	SDL_SetColours(s2, m_sysVideo->m_colours, 0, 256);
	SDL_BlitSurface(s, &rc, s2, &rcto);
	return s2;
}

SDL_Surface *JoyceMenuTerm::saveSurface(SDL_Rect &rc)
{
	return saveSurface(getSurface(), rc);
}


void JoyceMenuTerm::restoreSurface(SDL_Surface *s, SDL_Rect &rc)
{
	SDL_Surface *scr = getSurface();

        SDL_BlitSurface(s, NULL, scr, &rc);
        SDL_FreeSurface(s);
        SDL_UpdateRects(scr, 1, &rc);
}





void JoyceMenuTerm::centreContainer(UiContainer &c)
{
        unsigned int sx, sy, wc, hc;

	c.pack();
        sx = c.getWidth() / ui_char_w();
        sy = c.getHeight() / ui_char_h();

        wc = ui_screen_w() / ui_char_w();
        hc = ui_screen_h() / ui_char_h();

        c.setX(ui_char_w() * ((wc - sx) / 2));
        c.setY(ui_char_h() * ((hc - sy) / 2));
}



void JoyceMenuTerm::popup(const string message, int t, int centred)
{
        strncpy(m_popup, message.c_str(), 79); m_popup[79] = 0;

        if (t)
        {
                time(&m_popupTimeout);
                m_popupTimeout += t;
        }
        else    m_popupTimeout = 0;

        m_popupRect.w = 16 + ui_char_w() * strlen(m_popup);
	m_popupRect.x = (ui_screen_w() - m_popupRect.w) / 2;
        m_popupRect.h = 3 * ui_char_h();

        m_popupRect.y = ui_screen_h() - m_popupRect.h;
	if (centred) m_popupRect.y /= 2;

	m_popupBacking = saveSurface(m_sys->m_video->getSurface(), m_popupRect);
	m_popupContext = createPopup();
	drawPopup();
}



void JoyceMenuTerm::popupRedraw(const string message, int centred)
{
	char cmsg[80];
	
	memset(cmsg, ' ', 79);
	cmsg[79] = 0;
	if (message.length() > 79) memcpy(cmsg, message.c_str(), 79); 
	else
	{
		int x = (79 - message.length()) / 2;
		memcpy(cmsg + x, message.c_str(), message.length());
	}

	if (!m_popup[0])
	{
		popup(cmsg, 0, centred);
		return;	
	}
	strcpy(m_popup, cmsg);
	setPopupText(m_popupContext);
	drawPopup();
}


int JoyceMenuTerm::drawPopup(void)
{
	SDL_Rect rc;

	if (!m_popup[0]) return 0;

        memcpy(&rc, &m_popupRect, sizeof(SDL_Rect));
        rc.x = rc.y = 0;

        SDL_BlitSurface(m_popupContext, &rc, m_sys->m_video->getSurface(), &m_popupRect);
        SDL_UpdateRects(m_sys->m_video->getSurface(), 1, &m_popupRect);

        if (m_popupTimeout)
        {
                time_t tm;
                time (&tm);
                if (tm == m_popupTimeout) popupOff();
        	return 2;
	}
        return 1;
}

void JoyceMenuTerm::popupOff(void)
{
        SDL_Rect rc;
	if (!m_popup[0]) return;

        m_popupTimeout = 0;
        m_popup[0] = 0;

	memcpy(&rc, &m_popupRect, sizeof(SDL_Rect));
	rc.x = rc.y = 0;

        SDL_BlitSurface(m_popupBacking, &rc, m_sys->m_video->getSurface(), &m_popupRect);
	SDL_UpdateRects(m_sys->m_video->getSurface(), 1, &m_popupRect);
	SDL_FreeSurface(m_popupBacking);
	m_popupBacking = NULL;
}


void JoyceMenuTerm::onTimer50(void)
{
	m_term->onTimer50();
}

UiEvent JoyceMenuTerm::menuEvent(SDL_Event &e)
{
        int r = m_term->onEvent(e);

        switch(r)
        {
                case -2: return UIE_QUIT;
                case 0:  return UIE_CONTINUE;
                case 1:  return UIE_OK;
        }
        return UIE_CONTINUE;
}


char JoyceMenuTerm::getKey(void)
{
        if (!m_term->kbhit()) return 0;
        return m_term->read();
}


void JoyceMenuTerm::setManualUpdate(bool b)
{
	m_term->setManualUpdate(b);	
}

bool JoyceMenuTerm::getManualUpdate(void)
{
	return m_term->getManualUpdate();
}

void JoyceMenuTerm::updateRect(SDL_Rect &rc)
{
	SDL_UpdateRects(m_term->getSurface(), 1, &rc);
}


UiEvent JoyceMenuTerm::yesno(const string prompt)
{
	UiEvent uie;
	UiMenu mc(this);

	if (m_menuMutex) return UIE_CONTINUE;

	m_menuMutex = true;
        m_oldTerm = m_sys->m_activeTerminal;
	SDL_BlitSurface(m_sysVideo->getSurface(), NULL,
			m_framebuf->getSurface(), NULL);
        m_sys->selectTerminal(this);
	SDL_ShowCursor(1);

	mc.add(new UiLabel(prompt, this))
	  .add(new UiSeparator(this))
	  .add(new UiCommand(SDLK_y, "  Yes  ", this))
	  .add(new UiCommand(SDLK_n, "  No  ",  this));
	centreContainer(mc);
	uie = mc.eventLoop();

	m_sys->selectTerminal(m_oldTerm);
	m_menuMutex = 0;

	if (uie == UIE_QUIT) return uie;
	if (mc.getKey(mc.getSelected()) == SDLK_y) return UIE_OK;
	return UIE_CANCEL;
}




