/************************************************************************

    JOYCE v1.90 - Amstrad PCW emulator

    Copyright (C) 1996, 2001  John Elliott <jce@seasip.demon.co.uk>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*************************************************************************/

#include "Joyce.hxx"
#include "UiContainer.hxx"
#include "UiLabel.hxx"
#include "UiSeparator.hxx"
#include "UiCommand.hxx"
#include "UiSetting.hxx"
#include "UiMenu.hxx"
#include "UiTextInput.hxx"
#include "UiScrollingMenu.hxx"
#include "JoyceFileList.hxx"
#include "JoyceFileChooser.hxx"

JoycePrinter::JoycePrinter(char *portname) : JoyceDevice("printer", portname)
{
	string s = "JoycePrinter::JoycePrinter {";
	s += portname;
	s += "} [0]";
	XLOG(s.c_str());

	m_fp = NULL;
//	m_filename[0] = 0;	// [v1.9.1] You can't *do* that with STL 
				// strings!
	time(&m_tick);
	m_timeout = 30;
#if defined(WIN32)
	m_filename = "lpt1:";
#elif defined(UNIX)
	m_filename = "|lpr";
#else
	m_filename = "";
#endif
	m_pclose = false;
}

JoycePrinter::~JoycePrinter()
{
	close();
}


void JoycePrinter::reset(void)
{
	close();
}


void JoycePrinter::writeChar(char c)
{
	if (!isEnabled()) return;

	if (!m_fp) open();
	if (m_fp) fputc(c, m_fp);
	time(&m_tick);
}


bool JoycePrinter::isBusy(void)
{
	if (!isEnabled()) return true;

	if (!m_fp) open();
	if (m_fp) return false;
	return true;
}


void JoycePrinter::printerTick(void)
{
	time_t t2;

	time(&t2);
	if ((t2 - m_tick) > m_timeout)  close();
	else if (m_fp) fflush(m_fp);
}

bool JoycePrinter::open(void)
{
	if (m_filename == "") return false;

	if (m_filename[0] == '|')
	{
		m_fp = popen(m_filename.c_str() + 1, "w");
		if (!m_fp) return false;
		m_pclose = true;
		return true;
	}

	m_fp = fopen(m_filename.c_str(), "a");
	if (!m_fp) return false;
	return true;
}

void JoycePrinter::close()
{
	if (!m_fp) return;

	if (m_pclose) { pclose(m_fp); m_fp = NULL; return; }

	fclose(m_fp); 
	m_fp = NULL;
}

bool JoycePrinter::parseNode(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
	char *t;

	close();

	parseEnabled(doc, ns, cur);
	parseFilename(doc, ns, cur, "output", m_filename);
	t = (char *)xmlGetProp(cur, (xmlChar *)"timeout");
	if (t) 
	{
		m_timeout = atoi(t);
		free(t);
	}
	return true;	
}

bool JoycePrinter::storeNode(xmlDocPtr doc, xmlNsPtr ns, xmlNodePtr cur)
{
	char buf[20];

	storeEnabled(doc, ns, cur);
	sprintf(buf, "%d", m_timeout);
	xmlSetProp(cur, (xmlChar *)"timeout", (xmlChar *)buf);	

	storeFilename(doc, ns, cur, "output", m_filename);
	return true;
}

UiEvent JoycePrinter::getPipe(UiDrawer *d)
{
	UiMenu uim(d);
	UiEvent e;

	uim.add(new UiTextInput(SDLK_u, "  UNIX command: _______________________________  ", d));
	uim.add(new UiSeparator(d));
	uim.add(new UiCommand(SDLK_o, "  OK  ", d));

	d->centreContainer(uim);
	if (m_filename == "")          ((UiTextInput &)uim[0]).setText("");
	else if (m_filename[0] == '|') ((UiTextInput &)uim[0]).setText(m_filename.substr(1));	
	else			       ((UiTextInput &)uim[0]).setText("");

	e = uim.eventLoop();

	if (e == UIE_OK)
	{
		m_filename = "|" + ((UiTextInput &)uim[0]).getText();
	}
	return e;
}


UiEvent JoycePrinter::settings(UiDrawer *d)
{
	int x,y,sel;
	UiEvent uie;
	
	x = y = 0;
	sel = -1;
	while (1)
	{
		UiMenu uim(d);
		bool bEnabled;

		bEnabled = isEnabled();
		uim.add(new UiLabel(getTitle(), d));
		uim.add(new UiSeparator(d));
		uim.add(new UiSetting(SDLK_d, !bEnabled, "  Disconnected  ", d));
		uim.add(new UiSetting(SDLK_c,  bEnabled, "  Connected  ",    d));

		if (bEnabled)
		{
			string ostat = "  Output is to ";
			if (m_filename == "")          ostat += "file: ";	
			else if (m_filename[0] == '|') ostat += "pipe: " + displayName(m_filename.substr(1), 40);
			else			       ostat += "file: " + displayName(m_filename, 40);
			uim.add(new UiSeparator(d));
			uim.add(new UiLabel  (ostat, d)); 
			uim.add(new UiCommand(SDLK_f,  "  Output to File...  ", d, UIG_SUBMENU));
			uim.add(new UiCommand(SDLK_u,  "  Output to UNIX command... ",d, UIG_SUBMENU));
#ifdef WIN32
			uim.add(new UiCommand(SDLK_1,  "  Output to LPT1:  ", d));
			uim.add(new UiCommand(SDLK_2,  "  Output to LPT2:  ", d));
			uim.add(new UiCommand(SDLK_3,  "  Output to LPT3:  ", d));
			uim.add(new UiCommand(SDLK_4,  "  Output to LPT4:  ", d));
#endif
		}

		d->centreContainer(uim);
		uim.add(new UiSeparator(d));
		uim.add(new UiCommand(SDLK_ESCAPE,  "  EXIT  ", d));
		uim.setSelected(sel);

		uie =  uim.eventLoop();
	
		if (uie == UIE_QUIT) return uie;
		sel = uim.getSelected();	
		switch(uim.getKey(sel))
		{
// This is all very temporary and should really be subsumed into the menu
			case SDLK_f: 
				{
				UiEvent e;
				JoyceFileChooser f("  OK  ", d);
				if (m_filename != "" && m_filename[0] != '|') f.initDir(m_filename);
				e = f.eventLoop();
				if (e == UIE_QUIT) return e;
				if (e == UIE_OK) m_filename = f.getChoice();
				}
				break;
			case SDLK_1: m_filename = "LPT1:"; break;
			case SDLK_2: m_filename = "LPT2:"; break;
			case SDLK_3: m_filename = "LPT3:"; break;
			case SDLK_4: m_filename = "LPT4:"; break;
			case SDLK_d: enable(false); break;
			case SDLK_c: enable();      break;
			case SDLK_u: 
					uie = getPipe(d);
					if (uie == UIE_QUIT) return uie;
					break;

			case SDLK_ESCAPE: close();  return UIE_OK;
			default: break;
		}

	}
	return UIE_OK;
}

string JoycePrinter::getTitle(void)
{
	return "Parallel printer";
}









